"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteWorkspace = exports.updateDefaultWorkspace = exports.getAllWorkspaces = exports.createWorkspace = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const firestore_1 = require("firebase-admin/firestore");
const workspace_1 = require("../../../../types/workspace");
const zod_1 = require("zod");
const createWorkspace = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        const parsedBody = workspace_1.createWorkspaceSchema.parse(req.body);
        const { workspaceName, colorSet } = parsedBody;
        if (!(user === null || user === void 0 ? void 0 : user.companyId)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Company ID is required.",
            });
        }
        if ((user === null || user === void 0 ? void 0 : user.role) !== "superadmin" && (user === null || user === void 0 ? void 0 : user.role) !== "admin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. Only superadmin and admin can create workspaces.",
            });
        }
        // Step 1: Fetch company document
        const companyDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .get();
        if (!companyDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Company not found.",
            });
        }
        const companyData = companyDoc.data();
        const plan = companyData === null || companyData === void 0 ? void 0 : companyData.plan;
        // Step 2: Fetch plan document to check workspace limit
        const planDoc = yield firebase_admin_1.adminDb.collection("plans").doc(plan.name).get();
        if (!planDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Plan not found.",
            });
        }
        const planData = planDoc.data();
        const workspaceLimit = (planData === null || planData === void 0 ? void 0 : planData.workspaceLimit) || 1;
        // Step 3: Check current workspace usage
        const existingWorkspacesSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .where("companyId", "==", user.companyId)
            .get();
        const workspaceCount = existingWorkspacesSnapshot.size;
        if (workspaceCount >= workspaceLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Workspace limit reached. Your current plan (${plan.name}) allows up to ${workspaceLimit} workspaces.`,
            });
        }
        // Step 4: Create workspace document
        const workspaceRef = firebase_admin_1.adminDb.collection("workspaces").doc();
        const workspaceId = workspaceRef.id;
        // Step 5: Handle image upload (optional)
        const image = req.file; // Assuming the image is being sent as 'multipart/form-data'
        let imageUrl = null;
        if (image) {
            const imageExtension = image.originalname.split(".").pop();
            const imageName = `image.${imageExtension}`;
            const imagePath = `${user.companyId}/workspaces/${workspaceId}/${imageName}`;
            const file = firebase_admin_1.bucket.file(imagePath);
            yield file.save(image.buffer, {
                metadata: { contentType: image.mimetype },
            });
            yield file.makePublic();
            imageUrl = `https://storage.googleapis.com/${file.bucket.name}/${file.name}`;
        }
        // Step 6: Save workspace data
        yield workspaceRef.set({
            workspaceName,
            companyId: user.companyId,
            colorSet,
            imageUrl,
            createdBy: user.uid,
            createdAt: new Date().toISOString(),
        });
        // Step 7: Increment workspaceUsage in the company's usage field
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .update({
            "usage.workspaceUsage": firestore_1.FieldValue.increment(1),
        });
        // Step 8: Check and set defaultWorkspace for the user if not set
        const userDocRef = firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users")
            .doc(user.uid);
        const userDoc = yield userDocRef.get();
        if (!userDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "User not found.",
            });
        }
        const userData = userDoc.data();
        if (!(userData === null || userData === void 0 ? void 0 : userData.defaultWorkspace)) {
            yield userDocRef.update({
                defaultWorkspace: workspaceId,
            });
        }
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "Workspace created successfully.",
            data: {
                workspaceId,
                workspaceName,
                colorSet,
                imageUrl,
            },
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.createWorkspace = createWorkspace;
const getAllWorkspaces = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        if (!(user === null || user === void 0 ? void 0 : user.companyId)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Company ID is required.",
            });
        }
        if (!["superadmin", "admin", "viewer", "editor"].includes(user.role)) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. Only authorized roles can view workspaces.",
            });
        }
        const workspacesSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .where("companyId", "==", user.companyId)
            .get();
        if (workspacesSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No workspaces found for this company.",
            });
        }
        const workspaces = workspacesSnapshot.docs.map((doc) => (Object.assign({ id: doc.id }, doc.data())));
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Workspaces retrieved successfully",
            data: workspaces,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.getAllWorkspaces = getAllWorkspaces;
const updateDefaultWorkspace = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        const { workspaceId } = req.body; // Expecting new default workspace ID in the request body
        // Ensure the user is authenticated and has a company ID
        if (!(user === null || user === void 0 ? void 0 : user.companyId)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Company ID is required.",
            });
        }
        if (!(user === null || user === void 0 ? void 0 : user.id)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "User ID is required.",
            });
        }
        // Validate that the workspace exists and belongs to the user's company
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        // Check if the workspace belongs to the same company as the user
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "This workspace does not belong to your company.",
            });
        }
        // Update the default workspace for the user inside the company's 'users' subcollection
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users")
            .doc(user.id)
            .update({
            defaultWorkspace: workspaceId,
            updatedAt: new Date().toISOString(),
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Default workspace updated successfully.",
            data: {
                defaultWorkspace: workspaceId,
            },
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.updateDefaultWorkspace = updateDefaultWorkspace;
const deleteWorkspace = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        const { workspaceId } = req.params;
        if (!(user === null || user === void 0 ? void 0 : user.companyId)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Company ID is required.",
            });
        }
        if ((user === null || user === void 0 ? void 0 : user.role) !== "superadmin" && (user === null || user === void 0 ? void 0 : user.role) !== "admin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. Only superadmin and admin can delete workspaces.",
            });
        }
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "This workspace does not belong to your company.",
            });
        }
        const existingWorkspacesSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .where("companyId", "==", user.companyId)
            .get();
        const workspaceCount = existingWorkspacesSnapshot.size;
        if (workspaceCount === 1) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Deletion disabled. Your company must have at least one workspace.",
            });
        }
        // Step 1: Delete the workspace document
        yield firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId).delete();
        // Step 2: Decrement workspaceUsage in the company's usage field
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .update({
            "usage.workspaceUsage": firestore_1.FieldValue.increment(-1),
        });
        // Step 3: Update users' default workspace if set to the deleted one
        const usersSnapshot = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users")
            .where("defaultWorkspace", "==", workspaceId)
            .get();
        const batch = firebase_admin_1.adminDb.batch();
        usersSnapshot.forEach((userDoc) => {
            const userRef = userDoc.ref;
            batch.update(userRef, { defaultWorkspace: null });
        });
        yield batch.commit();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Workspace deleted successfully, and users' default workspace updated.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteWorkspace = deleteWorkspace;
