"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteAllFromTrash = exports.deleteFromTrash = exports.restoreAllFromTrash = exports.restoreFromTrash = exports.viewTrashedItems = exports.moveToTrash = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const zod_1 = require("zod");
const moveToTrashParamsSchema = zod_1.z.object({
    workspaceId: zod_1.z.string().min(1, "Workspace ID is required."),
});
const moveToTrashBodySchema = zod_1.z
    .object({
    rekamanFolderId: zod_1.z.string().optional(),
    rekamanId: zod_1.z.string().optional(),
})
    .refine((data) => data.rekamanFolderId || data.rekamanId, "Either Rekaman Folder ID or Rekaman ID must be provided.");
const moveToTrash = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { workspaceId } = moveToTrashParamsSchema.parse(req.params);
        const { rekamanFolderId, rekamanId } = moveToTrashBodySchema.parse(req.body);
        const user = req.user;
        const trashedAt = new Date().toISOString();
        const deletedBy = user.id;
        // Validate workspace
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        const isAdminOrSuperadmin = (user === null || user === void 0 ? void 0 : user.role) === "superadmin" || (user === null || user === void 0 ? void 0 : user.role) === "admin";
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied.",
            });
        }
        // Move folder to trash
        if (rekamanFolderId && !rekamanId) {
            const folderDoc = yield firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("rekamanFolders")
                .doc(rekamanFolderId)
                .get();
            if (!folderDoc.exists) {
                return res.status(404).json({
                    status: "error",
                    code: 404,
                    message: "Rekaman Folder not found.",
                });
            }
            const folderData = folderDoc.data();
            const isFolderCreator = (folderData === null || folderData === void 0 ? void 0 : folderData.createdBy) === user.id;
            if (!isAdminOrSuperadmin && !isFolderCreator) {
                return res.status(403).json({
                    status: "error",
                    code: 403,
                    message: "You are not allowed to move this Rekaman folder to trash.",
                });
            }
            // Find documents with matching divisiFolderId
            const documentsSnapshot = yield firebase_admin_1.adminDb
                .collection("rekaman")
                .where("rekamanFolderId", "==", rekamanFolderId)
                .get();
            // Create batch for moving the folder and associated documents
            const batch = firebase_admin_1.adminDb.batch();
            const trashedFolderRef = firebase_admin_1.adminDb
                .collection("trashedRekaman")
                .doc(rekamanFolderId);
            // Move folder data to trash
            batch.set(trashedFolderRef, Object.assign(Object.assign({}, folderData), { trashedAt,
                deletedBy, companyIdDeleted: user.companyId, originalWorkspaceId: workspaceId, originalRekamanFolderId: rekamanFolderId, type: "folder" }));
            // Move associated documents to sub-collection under trashed folder
            documentsSnapshot.forEach((doc) => {
                const documentData = doc.data();
                const trashedDocumentRef = trashedFolderRef
                    .collection("rekaman")
                    .doc(doc.id);
                batch.set(trashedDocumentRef, Object.assign(Object.assign({}, documentData), { trashedAt,
                    deletedBy, companyIdDeleted: user.companyId, originalWorkspaceId: workspaceId, originalRekamanFolderId: rekamanFolderId, type: "rekaman" }));
                // Delete the original document
                batch.delete(doc.ref);
            });
            // Delete the folder after moving it to trash
            batch.delete(folderDoc.ref);
            yield batch.commit();
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "Rekaman Folder and associated rekaman moved to trash successfully.",
            });
        }
        // Move individual document to trash (folderId + documentId provided)
        if (rekamanId && rekamanFolderId) {
            const folderRef = firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("rekamanFolders")
                .doc(rekamanFolderId);
            const folderSnapshot = yield folderRef.get();
            if (!folderSnapshot.exists) {
                return res.status(404).json({
                    status: "error",
                    code: 404,
                    message: "Rekaman Folder not found in the specified workspace.",
                });
            }
            const folderData = folderSnapshot.data();
            const hasFolderAccess = ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.allowedRoles.includes(user.role)) ||
                ((_b = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _b === void 0 ? void 0 : _b.assignedUsers.includes(user.id));
            if (!hasFolderAccess) {
                return res.status(403).json({
                    status: "error",
                    code: 403,
                    message: "You do not have permission to access this rekaman folder.",
                });
            }
            const documentDoc = yield firebase_admin_1.adminDb
                .collection("rekaman")
                .doc(rekamanId)
                .get();
            if (!documentDoc.exists) {
                return res.status(404).json({
                    status: "error",
                    code: 404,
                    message: "Rekaman not found.",
                });
            }
            const documentData = documentDoc.data();
            const isDocumentCreator = (documentData === null || documentData === void 0 ? void 0 : documentData.createdBy) === user.id;
            if (!isAdminOrSuperadmin && !isDocumentCreator) {
                return res.status(403).json({
                    status: "error",
                    code: 403,
                    message: "You are not allowed to move this rekaman to trash.",
                });
            }
            // Move document data to trash
            const trashedDocumentRef = firebase_admin_1.adminDb
                .collection("trashedRekaman")
                .doc(rekamanId); // Store individual document in the same collection
            yield trashedDocumentRef.set(Object.assign(Object.assign({}, documentData), { trashedAt,
                deletedBy, companyIdDeleted: user.companyId, originalWorkspaceId: workspaceId, originalRekamanFolderId: rekamanFolderId, type: "document" }));
            // Delete the original document
            yield documentDoc.ref.delete();
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "Rekaman moved to trash successfully.",
            });
        }
        return res.status(400).json({
            status: "error",
            code: 400,
            message: "Either rekaman folder ID or rekaman ID must be provided.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.moveToTrash = moveToTrash;
const restoreFromTrash = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { itemId } = req.params;
        const batch = firebase_admin_1.adminDb.batch();
        // Fetch the item from the trashedFolders collection
        const trashedItemDoc = yield firebase_admin_1.adminDb
            .collection("trashedFolders")
            .doc(itemId)
            .get();
        if (!trashedItemDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Item not found in trash.",
            });
        }
        const trashedItemData = trashedItemDoc.data();
        const type = trashedItemData === null || trashedItemData === void 0 ? void 0 : trashedItemData.type;
        const originalWorkspaceId = trashedItemData === null || trashedItemData === void 0 ? void 0 : trashedItemData.originalWorkspaceId;
        if (!type || !originalWorkspaceId) {
            return res.status(500).json({
                status: "error",
                code: 500,
                message: "Invalid trash data.",
            });
        }
        // Prepare the restored data by deleting fields like companyIdDeleted, trashedAt, etc.
        const _a = trashedItemData || {}, { trashedAt, originalRekamanFolderId, // Will be undefined if not present
        originalWorkspaceId: _unusedOriginalWorkspaceId, companyIdDeleted, // Remove this field
        deletedBy } = _a, restoredData = __rest(_a, ["trashedAt", "originalRekamanFolderId", "originalWorkspaceId", "companyIdDeleted", "deletedBy"]);
        // Restore folder
        if (type === "folder") {
            const folderRef = firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(originalWorkspaceId)
                .collection("folders")
                .doc(itemId);
            batch.set(folderRef, restoredData); // Restore without companyIdDeleted
            batch.delete(trashedItemDoc.ref); // Remove from trash
            // Restore associated documents
            const trashedDocumentsSnapshot = yield firebase_admin_1.adminDb
                .collection("trashedFolders")
                .doc(itemId)
                .collection("documents")
                .get();
            trashedDocumentsSnapshot.forEach((doc) => {
                const trashedDocumentData = doc.data();
                const { trashedAt, companyIdDeleted, // Remove this field from documents as well
                originalRekamanFolderId, originalWorkspaceId: _unusedOriginalWorkspaceId, deletedBy } = trashedDocumentData, restoredDocumentData = __rest(trashedDocumentData, ["trashedAt", "companyIdDeleted", "originalRekamanFolderId", "originalWorkspaceId", "deletedBy"]);
                const restoredDocumentRef = firebase_admin_1.adminDb.collection("documents").doc(doc.id);
                batch.set(restoredDocumentRef, restoredDocumentData);
                batch.delete(doc.ref);
            });
            yield batch.commit();
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "Folder and associated documents restored successfully.",
            });
        }
        // Restore document
        if (type === "document") {
            const restoredDocumentRef = firebase_admin_1.adminDb.collection("documents").doc(itemId);
            batch.set(restoredDocumentRef, restoredData); // Restore without companyIdDeleted
            batch.delete(trashedItemDoc.ref);
            yield batch.commit();
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "Document restored successfully.",
            });
        }
        return res.status(400).json({
            status: "error",
            code: 400,
            message: "Unknown item type in trash.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.restoreFromTrash = restoreFromTrash;
const restoreAllFromTrash = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const batch = firebase_admin_1.adminDb.batch();
        // Fetch all trashed items for the workspace
        const trashedItemsSnapshot = yield firebase_admin_1.adminDb
            .collection("trashedFolders")
            .where("originalWorkspaceId", "==", workspaceId)
            .get();
        if (trashedItemsSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No items found in trash.",
            });
        }
        // Create an array to hold promises for batch operations
        const batchOperations = [];
        // Iterate through all trashed items and restore them
        trashedItemsSnapshot.forEach((doc) => {
            const trashedItemData = doc.data();
            const type = trashedItemData === null || trashedItemData === void 0 ? void 0 : trashedItemData.type;
            const originalWorkspaceId = trashedItemData === null || trashedItemData === void 0 ? void 0 : trashedItemData.originalWorkspaceId;
            // Remove companyIdDeleted if present
            const { companyIdDeleted } = trashedItemData, // Remove this field
            restoredData = __rest(trashedItemData, ["companyIdDeleted"]);
            if (type === "folder") {
                const folderRef = firebase_admin_1.adminDb
                    .collection("workspaces")
                    .doc(originalWorkspaceId)
                    .collection("folders")
                    .doc(doc.id);
                batch.set(folderRef, restoredData); // Restore without companyIdDeleted
                batch.delete(doc.ref);
                // Restore associated documents
                const restoreDocuments = firebase_admin_1.adminDb
                    .collection("trashedFolders")
                    .doc(doc.id)
                    .collection("documents")
                    .get()
                    .then((trashedDocumentsSnapshot) => {
                    trashedDocumentsSnapshot.forEach((documentDoc) => {
                        const trashedDocumentData = documentDoc.data();
                        const { companyIdDeleted } = trashedDocumentData, // Remove this field from documents as well
                        restoredDocumentData = __rest(trashedDocumentData, ["companyIdDeleted"]);
                        const restoredDocumentRef = firebase_admin_1.adminDb
                            .collection("documents")
                            .doc(documentDoc.id);
                        batch.set(restoredDocumentRef, restoredDocumentData);
                        batch.delete(documentDoc.ref);
                    });
                });
                // Add to batch operations array
                batchOperations.push(restoreDocuments);
            }
            else if (type === "document") {
                const restoredDocumentRef = firebase_admin_1.adminDb.collection("documents").doc(doc.id);
                batch.set(restoredDocumentRef, restoredData); // Restore without companyIdDeleted
                batch.delete(doc.ref);
            }
        });
        // Wait for all document restore operations to complete
        yield Promise.all(batchOperations);
        // Commit the batch after all operations are added
        yield batch.commit();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "All items restored from trash successfully.",
        });
    }
    catch (error) {
        console.error(error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.restoreAllFromTrash = restoreAllFromTrash;
const viewTrashedItems = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const user = req.user;
        const isAdminOrSuperadmin = (user === null || user === void 0 ? void 0 : user.role) === "superadmin" || (user === null || user === void 0 ? void 0 : user.role) === "admin";
        // Fetch trashed folders and documents (both are stored in trashedFolders)
        const trashedFoldersSnap = yield firebase_admin_1.adminDb
            .collection("trashedFolders")
            .where("originalWorkspaceId", "==", workspaceId)
            .get();
        let trashedItems = trashedFoldersSnap.docs.map((doc) => doc.data());
        // Filter by deletedBy for non-admin users
        if (!isAdminOrSuperadmin) {
            trashedItems = trashedItems.filter((item) => item.deletedBy === user.id);
        }
        // Map through the trashed items (folders and documents) to include deletedBy and category details
        const trashedItemsWithDetails = yield Promise.all(trashedItems.map((item) => __awaiter(void 0, void 0, void 0, function* () {
            var _a;
            const { deletedBy, companyIdDeleted, categoryId, type } = item;
            // Fetch user who deleted the item
            const userDoc = yield firebase_admin_1.adminDb
                .collection(`companies/${companyIdDeleted}/users`)
                .doc(deletedBy)
                .get();
            const userData = userDoc.exists ? userDoc.data() : null;
            // Fetch category details for documents only
            let category = null;
            if (type === "document" && categoryId) {
                const categoryDoc = yield firebase_admin_1.adminDb
                    .collection("categories")
                    .doc(categoryId)
                    .get();
                category = categoryDoc.exists
                    ? { id: categoryId, categoryName: (_a = categoryDoc.data()) === null || _a === void 0 ? void 0 : _a.categoryName }
                    : { id: categoryId, categoryName: "Unknown" };
            }
            return Object.assign(Object.assign({}, item), { deletedBy: {
                    name: (userData === null || userData === void 0 ? void 0 : userData.name) || "Unknown",
                    email: (userData === null || userData === void 0 ? void 0 : userData.email) || "Unknown",
                }, category: category ||
                    (type === "document"
                        ? { id: null, categoryName: "No Category" }
                        : null) });
        })));
        return res.status(200).json({
            status: "success",
            code: 200,
            data: trashedItemsWithDetails,
        });
    }
    catch (error) {
        console.error(error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewTrashedItems = viewTrashedItems;
const deleteFromTrash = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { itemId, workspaceId } = req.params; // Get workspaceId from request parameters
        const batch = firebase_admin_1.adminDb.batch();
        // Fetch the trashed item document
        const trashedItemDoc = yield firebase_admin_1.adminDb
            .collection("trashedFolders")
            .doc(itemId)
            .get();
        if (!trashedItemDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Item not found in trash.",
            });
        }
        const trashedItemData = trashedItemDoc.data();
        const type = trashedItemData === null || trashedItemData === void 0 ? void 0 : trashedItemData.type;
        const originalWorkspaceId = trashedItemData === null || trashedItemData === void 0 ? void 0 : trashedItemData.originalWorkspaceId;
        // Check if the trashed item belongs to the same workspace as the user making the request
        if (originalWorkspaceId !== workspaceId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to delete this item from the trash.",
            });
        }
        if (!type || !originalWorkspaceId) {
            return res.status(500).json({
                status: "error",
                code: 500,
                message: "Invalid trash data.",
            });
        }
        // Delete folder and associated documents
        if (type === "folder") {
            // Also delete associated documents in the folder
            const trashedDocumentsSnapshot = yield firebase_admin_1.adminDb
                .collection("trashedFolders")
                .doc(itemId)
                .collection("documents")
                .get();
            trashedDocumentsSnapshot.forEach((doc) => {
                batch.delete(doc.ref); // Delete each document in the trashed folder
            });
            // Delete the folder itself from trash
            batch.delete(trashedItemDoc.ref);
        }
        // Delete single document
        if (type === "document") {
            // Delete the document itself from trash
            batch.delete(trashedItemDoc.ref);
        }
        // Commit the batch operation
        yield batch.commit();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: `${type === "folder" ? "Folder and associated documents" : "Document"} permanently deleted from trash.`,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteFromTrash = deleteFromTrash;
const deleteAllFromTrash = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const user = req.user; // Assuming this comes from your AuthenticatedRequest
        // Fetch all trashed items for the workspace
        const trashedItemsSnapshot = yield firebase_admin_1.adminDb
            .collection("trashedFolders")
            .where("originalWorkspaceId", "==", workspaceId)
            .get();
        if (trashedItemsSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No items found in trash.",
            });
        }
        // Check if user is superadmin, admin, or deleted the items
        const batch = firebase_admin_1.adminDb.batch();
        for (const doc of trashedItemsSnapshot.docs) {
            const trashedItemData = doc.data();
            const deletedBy = trashedItemData.deletedBy; // Assuming there's a 'deletedBy' field
            // Allow deletion if user is superadmin, admin, or if they deleted the item
            if (user.role === "superadmin" ||
                user.role === "admin" ||
                user.id === deletedBy) {
                const type = trashedItemData === null || trashedItemData === void 0 ? void 0 : trashedItemData.type;
                // Handle folder deletion and associated documents
                if (type === "folder") {
                    const trashedDocumentsSnapshot = yield firebase_admin_1.adminDb
                        .collection("trashedFolders")
                        .doc(doc.id)
                        .collection("documents")
                        .get();
                    trashedDocumentsSnapshot.forEach((documentDoc) => {
                        batch.delete(documentDoc.ref);
                    });
                }
                batch.delete(doc.ref);
            }
        }
        yield batch.commit();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "All items permanently deleted from trash.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteAllFromTrash = deleteAllFromTrash;
