"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteComment = exports.editComment = exports.addComment = exports.checkDeadlinesAndSendEmails = exports.submitTask = exports.viewTaskByPIC = exports.viewTaskByCreator = exports.deleteTask = exports.viewTask = exports.editTask = exports.createTask = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const zod_1 = require("zod");
const task_1 = require("../../../../types/task");
const email_1 = require("../../../../utils/email");
const notifications_1 = require("./notifications");
const firestore_1 = require("firebase-admin/firestore");
const uuid_1 = require("uuid");
const createEmailContent = ({ type, fromName, recipientName, message, link, }) => {
    let subject = "";
    let content = "";
    switch (type) {
        case "task":
            subject = `New Task Assigned by ${fromName}`;
            content = `Dear ${recipientName},\n\nYou have been assigned a new task. Please check the following link:\n\n${link}`;
            break;
        case "approval":
            subject = `Document Approval Request from ${fromName}`;
            content = `Dear ${recipientName},\n\n${message}\n\nPlease review the document using the link below:\n\n${link}`;
            break;
        case "rejection":
            subject = `Task Rejected by ${fromName}`;
            content = `Dear ${recipientName},\n\n${message}\n\nYou can view the task details using the link below:\n\n${link}`;
            break;
        default:
            break;
    }
    return { subject, content };
};
const validateWorkspaceAccess = (workspaceId, user) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Check if the workspace exists and is part of the user's company
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists) {
            return {
                isValid: false,
                code: 404,
                message: "Workspace not found",
            };
        }
        const workspaceData = workspaceDoc.data();
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return {
                isValid: false,
                code: 403,
                message: "Unauthorized access to this workspace",
            };
        }
        return {
            isValid: true,
            workspaceData, // Pass back workspace data if needed later
        };
    }
    catch (error) {
        return {
            isValid: false,
            code: 500,
            message: "Error validating workspace access",
        };
    }
});
const validatePICUser = (pic, companyId) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    const userDoc = yield firebase_admin_1.adminDb
        .collection("companies")
        .doc(companyId)
        .collection("users")
        .doc(pic)
        .get();
    if (!userDoc.exists) {
        return {
            isValid: false,
            message: "PIC user not found.",
        };
    }
    const userData = userDoc.data();
    if ((userData === null || userData === void 0 ? void 0 : userData.companyId) !== companyId) {
        return {
            isValid: false,
            message: "PIC user is not in the same company.",
        };
    }
    return {
        isValid: true,
        id: userData.id,
        companyId: userData.companyId,
        name: userData.name,
        email: userData.email,
        photoURL: userData.photoURL,
        taskNotification: (_b = (_a = userData === null || userData === void 0 ? void 0 : userData.settings) === null || _a === void 0 ? void 0 : _a.notifications) === null || _b === void 0 ? void 0 : _b.taskNotification,
    };
});
const createTask = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        const isAdminOrSuperadmin = () => (user === null || user === void 0 ? void 0 : user.role) === "superadmin" || (user === null || user === void 0 ? void 0 : user.role) === "admin";
        if (!isAdminOrSuperadmin()) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Only superadmins and admins can create tasks.",
            });
        }
        const parsedBody = task_1.createTaskSchema.parse(req.body);
        const { workspaceId } = req.params;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        // Validate PIC user is in the same company
        const picValidation = yield validatePICUser(parsedBody.pic, user.companyId);
        if (!picValidation.isValid) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: picValidation.message,
            });
        }
        // Validate workspace access
        const workspaceAccess = yield validateWorkspaceAccess(workspaceId, user);
        if (!workspaceAccess.isValid) {
            return res.status(workspaceAccess.code).json({
                status: "error",
                code: workspaceAccess.code,
                message: workspaceAccess.message,
            });
        }
        const taskRef = firebase_admin_1.adminDb.collection("tasks").doc();
        yield taskRef.set(Object.assign(Object.assign({}, parsedBody), { picEmail: picValidation.email, status: "pending", workspaceId, id: taskRef.id, companyId: user.companyId, createdAt: new Date().toISOString(), createdBy: user.id, requesterId: user.id }));
        if (picValidation.taskNotification) {
            yield (0, email_1.sendEmailTask)({
                toEmail: picValidation.email,
                fromName: user.name,
                recipientName: picValidation.name,
                link: `${process.env.APP_URL}/task/assigned-tasks?taskId=${taskRef.id}`, // Link to the task
            });
            yield (0, notifications_1.sendNotificationSingle)(taskRef.id, parsedBody.title, picValidation === null || picValidation === void 0 ? void 0 : picValidation.companyId, user.id, user.name, user.photoURL || undefined, picValidation === null || picValidation === void 0 ? void 0 : picValidation.id, picValidation === null || picValidation === void 0 ? void 0 : picValidation.name, picValidation === null || picValidation === void 0 ? void 0 : picValidation.email, (picValidation === null || picValidation === void 0 ? void 0 : picValidation.photoURL) || null, workspaceId, "task assign", `You have been assigned a new task titled "${parsedBody.title}" by ${user === null || user === void 0 ? void 0 : user.name}.`, "taskNotification");
        }
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "Task created successfully.",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.createTask = createTask;
const editTask = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        const { taskId, workspaceId } = req.params;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        if (!taskId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Task ID is required.",
            });
        }
        // Parse and validate request body using Zod
        const parsedBody = task_1.editTaskSchema.parse(req.body);
        const taskDoc = yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).get();
        const taskData = taskDoc.data();
        if (!taskDoc.exists || (taskData === null || taskData === void 0 ? void 0 : taskData.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Task not found or access denied.",
            });
        }
        // Check if the user is an admin or superadmin
        const isAdminOrSuperadmin = user.role === "superadmin" || user.role === "admin";
        if (!(isAdminOrSuperadmin ||
            taskData.createdBy === user.id ||
            taskData.pic === user.id)) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You are not authorized to edit this task.",
            });
        }
        // Validate PIC user if it exists in the request
        let picValidation = null;
        if (parsedBody.pic) {
            picValidation = yield validatePICUser(parsedBody.pic, user.companyId);
            if (!picValidation.isValid) {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: picValidation.message,
                });
            }
        }
        yield taskDoc.ref.update(Object.assign(Object.assign({}, parsedBody), { updatedAt: new Date().toISOString(), updatedBy: user.id }));
        picValidation = yield validatePICUser(taskData.pic, user.companyId);
        // Send notifications if the task status is updated to "approved" or "rejected"
        if (parsedBody.status && ["approved", "rejected"].includes(parsedBody.status)) {
            const notificationMessages = {
                approved: `The task "${taskData === null || taskData === void 0 ? void 0 : taskData.title}" has been approved.`,
                rejected: `The task "${taskData === null || taskData === void 0 ? void 0 : taskData.title}" has been rejected.`,
            };
            const notificationType = parsedBody.status === "approved" ? "task approved" : "task rejected";
            const message = notificationMessages[parsedBody.status];
            if (message && picValidation) {
                yield (0, notifications_1.sendNotificationSingle)(taskId, taskData.title, picValidation.companyId, user.id, user.name, user.photoURL || undefined, picValidation.id, picValidation.name, picValidation.email, picValidation.photoURL || null, workspaceId, notificationType, message, "taskNotification");
            }
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Task updated successfully.",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.editTask = editTask;
const viewTask = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f;
    try {
        const { taskId, workspaceId } = req.params;
        const user = req.user;
        if (!taskId || !workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Task ID and Workspace ID are required.",
            });
        }
        const taskDoc = yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).get();
        if (!taskDoc.exists) {
            return res.status(200).json({
                status: "success",
                code: 200,
                data: [],
            });
        }
        const taskData = taskDoc.data();
        if ((taskData === null || taskData === void 0 ? void 0 : taskData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied.",
            });
        }
        const isAdminOrSuperadmin = user.role === "superadmin" || user.role === "admin";
        const isTaskPIC = user.id === (taskData === null || taskData === void 0 ? void 0 : taskData.pic);
        if (!isAdminOrSuperadmin &&
            !isTaskPIC &&
            (taskData === null || taskData === void 0 ? void 0 : taskData.workspaceId) !== workspaceId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to access this task.",
            });
        }
        const [picUserDoc, creatorUserDoc] = yield Promise.all([
            firebase_admin_1.adminDb
                .collection("companies")
                .doc(taskData.companyId)
                .collection("users")
                .doc(taskData.pic)
                .get(),
            firebase_admin_1.adminDb
                .collection("companies")
                .doc(taskData.companyId)
                .collection("users")
                .doc(taskData.createdBy)
                .get(),
        ]);
        const picUser = picUserDoc.exists
            ? {
                id: picUserDoc.id,
                name: (_a = picUserDoc.data()) === null || _a === void 0 ? void 0 : _a.name,
                email: (_b = picUserDoc.data()) === null || _b === void 0 ? void 0 : _b.email,
                photoURL: ((_c = picUserDoc.data()) === null || _c === void 0 ? void 0 : _c.photoURL) || null,
            }
            : { id: taskData.pic, name: "Unknown User", email: "N/A", photoURL: null };
        const creatorUser = creatorUserDoc.exists
            ? {
                id: creatorUserDoc.id,
                name: (_d = creatorUserDoc.data()) === null || _d === void 0 ? void 0 : _d.name,
                email: (_e = creatorUserDoc.data()) === null || _e === void 0 ? void 0 : _e.email,
                photoURL: ((_f = creatorUserDoc.data()) === null || _f === void 0 ? void 0 : _f.photoURL) || null,
            }
            : { id: taskData.createdBy, name: "Unknown User", email: "N/A", photoURL: null };
        const commentsWithDetails = yield Promise.all((taskData.comments || []).map((comment) => __awaiter(void 0, void 0, void 0, function* () {
            var _a, _b, _c;
            if (!comment.userId)
                return comment; // Skip if no userId field
            const commentedByUserDoc = yield firebase_admin_1.adminDb
                .collection("companies")
                .doc(taskData.companyId)
                .collection("users")
                .doc(comment.userId)
                .get();
            const commentedByUser = commentedByUserDoc.exists
                ? {
                    id: commentedByUserDoc.id,
                    name: (_a = commentedByUserDoc.data()) === null || _a === void 0 ? void 0 : _a.name,
                    email: (_b = commentedByUserDoc.data()) === null || _b === void 0 ? void 0 : _b.email,
                    photoURL: ((_c = commentedByUserDoc.data()) === null || _c === void 0 ? void 0 : _c.photoURL) || null,
                }
                : { id: comment.userId, name: "Unknown User", email: "N/A", photoURL: null };
            return Object.assign(Object.assign({}, comment), { commentedByUser });
        })));
        return res.status(200).json({
            status: "success",
            code: 200,
            data: Object.assign(Object.assign({}, taskData), { assigner: creatorUser, assignee: picUser, comments: commentsWithDetails }),
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewTask = viewTask;
const viewTaskByPIC = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        // Query all tasks where the current user is the PIC
        const tasksQuery = yield firebase_admin_1.adminDb
            .collection("tasks")
            .where("pic", "==", user.id)
            .get();
        if (tasksQuery.empty) {
            return res.status(200).json({
                status: "success",
                code: 200,
                data: [],
            });
        }
        // Process tasks in parallel
        const tasks = yield Promise.all(tasksQuery.docs.map((taskDoc) => __awaiter(void 0, void 0, void 0, function* () {
            var _a, _b, _c, _d, _e, _f;
            const taskData = taskDoc.data();
            // Parallel fetching of category, creator, and PIC details
            const [picUserDoc, creatorUserDoc] = yield Promise.all([
                firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(taskData.companyId)
                    .collection("users")
                    .doc(taskData.pic)
                    .get(),
                firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(taskData.companyId)
                    .collection("users")
                    .doc(taskData.createdBy)
                    .get(),
            ]);
            const picUser = picUserDoc.exists
                ? {
                    id: picUserDoc.id,
                    name: (_a = picUserDoc.data()) === null || _a === void 0 ? void 0 : _a.name,
                    email: (_b = picUserDoc.data()) === null || _b === void 0 ? void 0 : _b.email,
                    photoURL: ((_c = picUserDoc.data()) === null || _c === void 0 ? void 0 : _c.photoURL) || null,
                }
                : { id: taskData.pic, name: "Unknown User", email: "N/A", photoURL: null };
            const creatorUser = creatorUserDoc.exists
                ? {
                    id: creatorUserDoc.id,
                    name: (_d = creatorUserDoc.data()) === null || _d === void 0 ? void 0 : _d.name,
                    email: (_e = creatorUserDoc.data()) === null || _e === void 0 ? void 0 : _e.email,
                    photoURL: ((_f = creatorUserDoc.data()) === null || _f === void 0 ? void 0 : _f.photoURL) || null,
                }
                : { id: taskData.createdBy, name: "Unknown User", email: "N/A", photoURL: null };
            // Add user details to notes
            const commentsWithDetails = yield Promise.all((taskData.comments || []).map((comment) => __awaiter(void 0, void 0, void 0, function* () {
                var _a, _b, _c;
                if (!comment.userId)
                    return comment; // Skip if no rejectedBy field
                const rejectedByUserDoc = yield firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(taskData.companyId)
                    .collection("users")
                    .doc(comment.userId)
                    .get();
                const commentedByUser = rejectedByUserDoc.exists
                    ? {
                        id: rejectedByUserDoc.id,
                        name: (_a = rejectedByUserDoc.data()) === null || _a === void 0 ? void 0 : _a.name,
                        email: (_b = rejectedByUserDoc.data()) === null || _b === void 0 ? void 0 : _b.email,
                        photoURL: ((_c = rejectedByUserDoc.data()) === null || _c === void 0 ? void 0 : _c.photoURL) || null,
                    }
                    : { id: comment.rejectedBy, name: "Unknown User", email: "N/A", photoURL: null };
                return Object.assign(Object.assign({}, comment), { commentedByUser });
            })));
            return Object.assign(Object.assign({}, taskData), { assigner: creatorUser, assignee: picUser, comments: commentsWithDetails });
        })));
        return res.status(200).json({
            status: "success",
            code: 200,
            data: tasks,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewTaskByPIC = viewTaskByPIC;
const viewTaskByCreator = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        // Query all tasks where the current user is the creator
        const tasksQuery = yield firebase_admin_1.adminDb
            .collection("tasks")
            .where("createdBy", "==", user.id)
            .get();
        if (tasksQuery.empty) {
            return res.status(200).json({
                status: "success",
                code: 200,
                data: [],
            });
        }
        // Process tasks in parallel
        const tasks = yield Promise.all(tasksQuery.docs.map((taskDoc) => __awaiter(void 0, void 0, void 0, function* () {
            var _a, _b, _c, _d, _e, _f;
            const taskData = taskDoc.data();
            // Parallel fetching of category, creator, and PIC details
            const [picUserDoc, creatorUserDoc] = yield Promise.all([
                firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(taskData.companyId)
                    .collection("users")
                    .doc(taskData.pic)
                    .get(),
                firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(taskData.companyId)
                    .collection("users")
                    .doc(taskData.createdBy)
                    .get(),
            ]);
            const picUser = picUserDoc.exists
                ? {
                    id: picUserDoc.id,
                    name: (_a = picUserDoc.data()) === null || _a === void 0 ? void 0 : _a.name,
                    email: (_b = picUserDoc.data()) === null || _b === void 0 ? void 0 : _b.email,
                    photoURL: ((_c = picUserDoc.data()) === null || _c === void 0 ? void 0 : _c.photoURL) || null,
                }
                : { id: taskData.pic, name: "Unknown User", email: "N/A", photoURL: null };
            const creatorUser = creatorUserDoc.exists
                ? {
                    id: creatorUserDoc.id,
                    name: (_d = creatorUserDoc.data()) === null || _d === void 0 ? void 0 : _d.name,
                    email: (_e = creatorUserDoc.data()) === null || _e === void 0 ? void 0 : _e.email,
                    photoURL: ((_f = creatorUserDoc.data()) === null || _f === void 0 ? void 0 : _f.photoURL) || null,
                }
                : { id: taskData.createdBy, name: "Unknown User", email: "N/A", photoURL: null };
            const commentsWithDetails = yield Promise.all((taskData.comments || []).map((comment) => __awaiter(void 0, void 0, void 0, function* () {
                var _a, _b, _c;
                if (!comment.userId)
                    return comment; // Skip if no userId field
                const commentedByUserDoc = yield firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(taskData.companyId)
                    .collection("users")
                    .doc(comment.userId)
                    .get();
                const commentedByUser = commentedByUserDoc.exists
                    ? {
                        id: commentedByUserDoc.id,
                        name: (_a = commentedByUserDoc.data()) === null || _a === void 0 ? void 0 : _a.name,
                        email: (_b = commentedByUserDoc.data()) === null || _b === void 0 ? void 0 : _b.email,
                        photoURL: ((_c = commentedByUserDoc.data()) === null || _c === void 0 ? void 0 : _c.photoURL) || null,
                    }
                    : { id: comment.rejectedBy, name: "Unknown User", email: "N/A", photoURL: null };
                return Object.assign(Object.assign({}, comment), { commentedByUser });
            })));
            return Object.assign(Object.assign({}, taskData), { assigner: creatorUser, assignee: picUser, comments: commentsWithDetails });
        })));
        return res.status(200).json({
            status: "success",
            code: 200,
            data: tasks,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewTaskByCreator = viewTaskByCreator;
const deleteTask = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        const { taskId, workspaceId } = req.params;
        if (!taskId || !workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Task ID and Workspace ID are required.",
            });
        }
        const taskDoc = yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).get();
        const taskData = taskDoc.data();
        if (!taskDoc.exists || (taskData === null || taskData === void 0 ? void 0 : taskData.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Task not found or access denied.",
            });
        }
        // Only superadmin or admin can delete the task
        if (user.role !== "superadmin" && user.role !== "admin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Only superadmin or admin can delete this task.",
            });
        }
        yield taskDoc.ref.delete();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Task deleted successfully.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteTask = deleteTask;
const submitTask = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { taskId } = req.params;
        const { user } = req;
        const { urlDocument } = req.body;
        // Validate the input
        if (!urlDocument) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "urlDocument is required.",
            });
        }
        // Fetch the task
        const taskDoc = yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).get();
        if (!taskDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Task not found.",
            });
        }
        const taskData = taskDoc.data();
        // Ensure the user is the PIC
        if (taskData.pic !== user.id) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to submit this task.",
            });
        }
        // Update the task with the document URL and status
        yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).update({
            status: "in review",
            submittedAt: new Date().toISOString(),
            urlDocument,
        });
        // Fetch the creator details
        const creatorDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users")
            .doc(taskData.createdBy)
            .get();
        if (!creatorDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Task creator not found.",
            });
        }
        const creatorUser = creatorDoc.data();
        if ((_b = (_a = creatorUser === null || creatorUser === void 0 ? void 0 : creatorUser.settings) === null || _a === void 0 ? void 0 : _a.notifications) === null || _b === void 0 ? void 0 : _b.taskNotification) {
            try {
                yield (0, notifications_1.sendNotificationSingle)(taskId, taskData.title, creatorUser === null || creatorUser === void 0 ? void 0 : creatorUser.companyId, user === null || user === void 0 ? void 0 : user.id, user === null || user === void 0 ? void 0 : user.name, (user === null || user === void 0 ? void 0 : user.photoURL) || undefined, creatorUser.id, creatorUser.name, creatorUser.email, creatorUser.photoURL || undefined, taskData.workspaceId, "task submitted", `The task "${taskData.title}" has been submitted by ${user === null || user === void 0 ? void 0 : user.name}.`, "taskNotification");
            }
            catch (notificationError) {
                throw new Error(`Notification error: ${notificationError.message}`);
            }
        }
        // Prepare and send email
        try {
            const emailContent = createEmailContent({
                type: "task",
                fromName: user.name,
                recipientName: creatorUser.name, // Ensure the creator's name exists
                link: `${process.env.APP_URL}/task/assigned-tasks?taskId=${taskId}`, // Replace with your app's task URL
            });
            yield (0, email_1.sendEmail)({
                toEmail: creatorUser.email, // Ensure the creator's email exists
                subject: emailContent.subject,
                content: emailContent.content,
                isHtml: false,
            });
        }
        catch (emailError) {
            throw new Error(`Email error: ${emailError.message}`);
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Task submitted for review successfully.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: error.message || "Internal Server Error",
        });
    }
});
exports.submitTask = submitTask;
const addComment = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        const { taskId } = req.params;
        const { content } = req.body;
        if (!content) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Content is required.",
            });
        }
        const taskDoc = yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).get();
        if (!taskDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Task not found.",
            });
        }
        const commentId = (0, uuid_1.v4)();
        const newComment = {
            id: commentId,
            content,
            userId: user.id,
            postedAt: new Date().toISOString(),
        };
        yield taskDoc.ref.update({
            comments: firestore_1.FieldValue.arrayUnion(newComment),
        });
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "Comment added successfully.",
            data: Object.assign(Object.assign({}, newComment), { commentedByUser: {
                    id: user.id,
                    name: user.name,
                    email: user.email,
                    photoURL: user.photoURL
                } }),
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.addComment = addComment;
const editComment = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        const { taskId, commentId } = req.params;
        const { content } = req.body;
        if (!content) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Content is required.",
            });
        }
        const taskDoc = yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).get();
        if (!taskDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Task not found.",
            });
        }
        const taskData = taskDoc.data();
        const comments = (taskData === null || taskData === void 0 ? void 0 : taskData.comments) || [];
        const commentIndex = comments.findIndex((c) => c.id === commentId);
        if (commentIndex === -1) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Comment not found.",
            });
        }
        if (comments[commentIndex].userId !== user.id) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You are not authorized to edit this comment.",
            });
        }
        const updatedComment = Object.assign(Object.assign({}, comments[commentIndex]), { content, updatedAt: new Date().toISOString() });
        comments[commentIndex] = updatedComment;
        yield taskDoc.ref.update({ comments });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Comment updated successfully.",
            data: Object.assign(Object.assign({}, updatedComment), { commentedByUser: {
                    id: user.id,
                    name: user.name,
                    email: user.email,
                    photoURL: user.photoURL,
                } }),
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.editComment = editComment;
const deleteComment = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        const { taskId, commentId } = req.params;
        const taskDoc = yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).get();
        if (!taskDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Task not found.",
            });
        }
        const taskData = taskDoc.data();
        const comments = (taskData === null || taskData === void 0 ? void 0 : taskData.comments) || [];
        const commentIndex = comments.findIndex((c) => c.id === commentId);
        if (commentIndex === -1) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Comment not found.",
            });
        }
        if (comments[commentIndex].userId !== user.id &&
            user.role !== "admin" &&
            user.role !== "superadmin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You are not authorized to delete this comment.",
            });
        }
        comments.splice(commentIndex, 1);
        yield taskDoc.ref.update({ comments });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Comment deleted successfully.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteComment = deleteComment;
const checkDeadlinesAndSendEmails = () => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const now = new Date();
        const todayStart = new Date(now.getFullYear(), now.getMonth(), now.getDate());
        const todayEnd = new Date(todayStart.getTime() + 24 * 60 * 60 * 1000 - 1); // End of today
        const oneWeekLater = new Date(now.getTime() + 7 * 24 * 60 * 60 * 1000);
        const oneDayLater = new Date(now.getTime() + 24 * 60 * 60 * 1000);
        const tasksSnapshot = yield firebase_admin_1.adminDb.collection("tasks").get();
        tasksSnapshot.forEach((taskDoc) => __awaiter(void 0, void 0, void 0, function* () {
            const task = taskDoc.data();
            const taskDeadline = new Date(task.deadline);
            if (task.status === "approved") {
                console.log(`Skipping task ${task.id} as it is already approved approved.`);
                return;
            }
            // Default notificationsEmail array if it doesn't exist
            const notificationsEmail = task.notificationsEmail || [];
            if (task.status.toLowerCase() !== "overdue") {
                // Check if the task is past its deadline
                if (taskDeadline < now && !notificationsEmail.includes("overdue")) {
                    yield updateTaskStatusToOverdue(taskDoc.id);
                    yield sendTaskReminderEmail(task, "overdue");
                    yield addNotificationSent(taskDoc.id, "overdue");
                }
                // Check if the deadline is today
                if (taskDeadline >= todayStart && taskDeadline <= todayEnd && !notificationsEmail.includes("today")) {
                    yield sendTaskReminderEmail(task, "today");
                    yield addNotificationSent(taskDoc.id, "today");
                }
                // Check if the deadline is less than a week away
                if (taskDeadline > todayEnd && taskDeadline <= oneWeekLater && !notificationsEmail.includes("1-week")) {
                    yield sendTaskReminderEmail(task, "1-week");
                    yield addNotificationSent(taskDoc.id, "1-week");
                }
                // Check if the deadline is one day away
                if (taskDeadline > todayEnd && taskDeadline <= oneDayLater && !notificationsEmail.includes("1-day")) {
                    yield sendTaskReminderEmail(task, "1-day");
                    yield addNotificationSent(taskDoc.id, "1-day");
                }
            }
        }));
    }
    catch (error) {
        console.error("Error checking deadlines:", error);
    }
});
exports.checkDeadlinesAndSendEmails = checkDeadlinesAndSendEmails;
const updateTaskStatusToOverdue = (taskId) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).update({ status: "overdue" });
        console.log(`Updated task ${taskId} to "overdue".`);
    }
    catch (error) {
        console.error(`Error updating task ${taskId} to "overdue":`, error);
    }
});
const addNotificationSent = (taskId, reminderType) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        yield firebase_admin_1.adminDb.collection("tasks").doc(taskId).update({
            notificationsEmail: firestore_1.FieldValue.arrayUnion(reminderType),
        });
        console.log(`Notification "${reminderType}" marked as sent for task ${taskId}.`);
    }
    catch (error) {
        console.error(`Error updating notificationsEmail for task ${taskId}:`, error);
    }
});
const sendTaskReminderEmail = (task, reminderType) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const recipientEmail = (task === null || task === void 0 ? void 0 : task.picEmail) || "fallback-email@example.com";
        const link = `${process.env.APP_URL}/task/assigned-tasks?taskId=${task.id}`;
        let subject = "";
        let message = "";
        // Define subject and message based on reminderType
        if (reminderType === "1-week") {
            subject = `Reminder: Task "${task.title}" is due in 1 week`;
            message = `Your task "${task.title}" is due on ${task.deadline}. Please ensure it is completed on time.\n\nYou can view the task details here: ${link}`;
        }
        else if (reminderType === "1-day") {
            subject = `Reminder: Task "${task.title}" is due tomorrow`;
            message = `Your task "${task.title}" is due tomorrow (${task.deadline}). Please ensure it is completed on time.\n\nYou can view the task details here: ${link}`;
        }
        else if (reminderType === "today") {
            subject = `Reminder: Task "${task.title}" is due today`;
            message = `Your task "${task.title}" is due today (${task.deadline}). Please ensure it is completed on time.\n\nYou can view the task details here: ${link}`;
        }
        else if (reminderType === "overdue") {
            subject = `Alert: Task "${task.title}" is overdue`;
            message = `Your task "${task.title}" is overdue since ${task.deadline}. Please complete it as soon as possible.\n\nYou can view the task details here: ${link}`;
        }
        // Send email
        yield (0, email_1.sendEmail)({
            toEmail: recipientEmail,
            subject,
            content: message,
            isHtml: false,
        });
        console.log(`Email sent to ${recipientEmail} with subject: "${subject}"`);
    }
    catch (error) {
        console.error(`Error sending email to ${task === null || task === void 0 ? void 0 : task.picEmail}:`, error);
    }
});
