"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteRekamanFolder = exports.viewRekamanFolder = exports.viewAllRekamanFolders = exports.editRekamanFolder = exports.createRekamanFolder = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const trash_1 = require("./trash");
const divisiFolder_1 = require("../../../../types/divisiFolder");
const zod_1 = require("zod");
const firestore_1 = require("firebase-admin/firestore");
const notifications_1 = require("./notifications");
const firestore_2 = require("firebase/firestore");
const firebase_1 = require("../../../../utils/firebase");
const typesense_1 = __importDefault(require("../../../../utils/typesense"));
const createRekamanFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const parsedBody = divisiFolder_1.createFolderSchema.parse(req.body);
        const { folderName, description, invitedUsers = [], parentFolderId = null, } = parsedBody;
        const user = req.user;
        // Check if the user is authenticated and part of a company
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to perform this action.",
            });
        }
        // Verify if the workspace exists
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        // Ensure the workspace belongs to the user's company
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You are not allowed to create rekaman folders in this workspace.",
            });
        }
        // Base folder collection reference
        const folderCollectionRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("rekamanFolders");
        // Check if the parent folder exists (if provided)
        if (parentFolderId) {
            const parentFolderDoc = yield folderCollectionRef
                .doc(parentFolderId)
                .get();
            if (!parentFolderDoc.exists) {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: "Parent folder not found.",
                });
            }
        }
        // Check if a folder with the same name already exists under the same parent folder
        const existingFolderSnapshot = yield folderCollectionRef
            .where("workspaceId", "==", workspaceId)
            .where("folderName", "==", folderName)
            .where("parentFolderId", "==", parentFolderId)
            .limit(1)
            .get();
        if (!existingFolderSnapshot.empty) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "A folder with the same name already exists.",
            });
        }
        // Ensure validInvitedUsers is correctly maintained
        const validInvitedUsers = [];
        if (invitedUsers.length > 0) {
            // Batch fetch user documents only if there are invited users
            const userDocs = yield firebase_admin_1.adminDb.getAll(...invitedUsers.map((invitedUser) => {
                var _a;
                return firebase_admin_1.adminDb
                    .collection("companies")
                    .doc((_a = req === null || req === void 0 ? void 0 : req.user) === null || _a === void 0 ? void 0 : _a.companyId)
                    .collection("users")
                    .doc(invitedUser.id);
            }));
            for (let i = 0; i < userDocs.length; i++) {
                const userDoc = userDocs[i];
                const invitedUser = invitedUsers[i];
                if (!userDoc.exists) {
                    return res.status(400).json({
                        status: "error",
                        code: 400,
                        message: `User with ID ${invitedUser.id} does not exist within this company.`,
                    });
                }
                const invitedUserData = userDoc.data();
                // Ensure invited users belong to the same company
                if ((invitedUserData === null || invitedUserData === void 0 ? void 0 : invitedUserData.companyId) !== user.companyId) {
                    return res.status(400).json({
                        status: "error",
                        code: 400,
                        message: `User with ID ${invitedUser.id} is not in the same company.`,
                    });
                }
                validInvitedUsers.push({
                    id: invitedUser.id,
                    email: (invitedUserData === null || invitedUserData === void 0 ? void 0 : invitedUserData.email) || null,
                    name: (invitedUserData === null || invitedUserData === void 0 ? void 0 : invitedUserData.name) || null,
                    photoURL: (invitedUserData === null || invitedUserData === void 0 ? void 0 : invitedUserData.photoURL) || null,
                    role: invitedUser.role,
                });
            }
        }
        // Estimate folder size (name + description overhead) in bytes
        const folderSizeBytes = Buffer.byteLength(folderName, "utf-8") +
            Buffer.byteLength(description || "", "utf-8");
        // Convert bytes to MB
        const folderSizeMB = folderSizeBytes / (1024 * 1024);
        // Fetch company data for storage usage validation
        const companyDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .get();
        const companyData = companyDoc.data();
        if (!companyData) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Company not found.",
            });
        }
        const { storageLimit = 0 } = companyData.plan; // Storage limit in MB
        const { usage = {} } = companyData; // Usage field (could be empty initially)
        const currentStorageUsage = usage.storageUsage || 0; // Storage usage in MB
        // Check if the folder creation exceeds storage limit
        if (currentStorageUsage + folderSizeMB > storageLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Allowed: ${storageLimit} MB, Used: ${currentStorageUsage} MB.`,
            });
        }
        // Create the new folder in the folders collection
        const folderRef = folderCollectionRef.doc(); // Automatically generate a new ID
        const folderId = folderRef.id;
        const folderNameInsensitive = folderName.toLowerCase();
        const folderData = {
            id: folderId,
            folderName,
            folderName_insensitive: folderNameInsensitive,
            description,
            parentFolderId: parentFolderId || null, // Store parent folder info if applicable
            workspaceId, // Add workspaceId to associate the folder with the workspace
            createdAt: new Date().toISOString(),
            createdBy: user.id,
            access: {
                allowedRoles: ["superadmin", "admin"],
                assignedUsers: validInvitedUsers,
                assignedUsersId: validInvitedUsers.map((user) => user.id),
                requiresApproval: true,
                ownerId: user.id,
            },
        };
        yield folderRef.set(folderData);
        const searchRekamanFolderRef = firebase_admin_1.adminDb
            .collection("searchableRekamanFolder")
            .doc(folderId);
        yield searchRekamanFolderRef.set(Object.assign(Object.assign({}, folderData), { companyId: user.companyId }));
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .update({
            "usage.storageUsage": firestore_1.FieldValue.increment(folderSizeMB), // Increment storage usage
        });
        const notificationRecipients = validInvitedUsers.map((user) => ({
            id: user.id,
            email: (user === null || user === void 0 ? void 0 : user.email) || "",
            name: (user === null || user === void 0 ? void 0 : user.name) || "",
            photoURL: user.photoURL || undefined,
            isRead: false, // Default to unread
        }));
        if (validInvitedUsers.length > 0) {
            yield (0, notifications_1.sendNotificationFolderAccess)(folderName, workspaceId, user.id, user.name, user.photoURL || undefined, folderId, "recording", `You have been granted access to the folder "${folderName}".`, notificationRecipients);
        }
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "Rekaman Folder created successfully",
            invitedUsers: validInvitedUsers,
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        // Catch unexpected errors
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.createRekamanFolder = createRekamanFolder;
function mergeAssignedUsers(currentAssignedUsers, newInvitedUsers, explicitRevokedUsers, companyId) {
    return __awaiter(this, void 0, void 0, function* () {
        const userMap = {};
        // Fetch user data from Firestore for new invited users
        const userPromises = newInvitedUsers.map((user) => __awaiter(this, void 0, void 0, function* () {
            const userDoc = yield firebase_admin_1.adminDb
                .collection("companies")
                .doc(companyId)
                .collection("users")
                .doc(user.id)
                .get();
            if (userDoc.exists) {
                const userData = userDoc.data();
                return {
                    id: user.id,
                    email: (userData === null || userData === void 0 ? void 0 : userData.email) || null,
                    name: (userData === null || userData === void 0 ? void 0 : userData.name) || null,
                    photoURL: (userData === null || userData === void 0 ? void 0 : userData.photoURL) || null,
                    role: user.role, // Assign role from new invited user data
                };
            }
            return null;
        }));
        // Add new invited users to the map
        const fetchedInvitedUsers = (yield Promise.all(userPromises)).filter(Boolean);
        fetchedInvitedUsers.forEach((user) => {
            userMap[user === null || user === void 0 ? void 0 : user.id] = user;
        });
        // Add current assigned users to the map, updating roles if necessary
        currentAssignedUsers.forEach((user) => {
            const isRevoked = explicitRevokedUsers.some((revokedUser) => revokedUser.id === user.id);
            if (!isRevoked) {
                // Check if the user exists in `newInvitedUsers` to update their role if needed
                const invitedUserWithNewRole = newInvitedUsers.find((invitedUser) => invitedUser.id === user.id);
                if (invitedUserWithNewRole) {
                    user.role = invitedUserWithNewRole.role; // Update role
                }
                userMap[user.id] = Object.assign(Object.assign({}, userMap[user.id]), user);
            }
        });
        // Collect updated assigned users without duplicates and excluding revoked users
        return Object.values(userMap);
    });
}
// const grantFolderAccess = async (
//   workspaceRef: FirebaseFirestore.DocumentReference,
//   folderId: string,
//   userId: string,
//   email: string,
//   name: string,
//   role: string,
//   photoURL: string,
//   folderType: string = "folders"
// ) => {
//   const addedUsers: Array<{
//     id: string;
//     email: string;
//     name: string;
//     photoURL?: string;
//     isRead: boolean;
//   }> = [];
//   while (folderId) {
//     const folderRef = workspaceRef.collection(folderType).doc(folderId);
//     const folderSnapshot = await folderRef.get();
//     const searchableRekamanFolderRef = adminDb
//       .collection("searchableRekamanFolder")
//       .doc(folderId);
//     const searchableRekamanFolderSnapshot =
//       await searchableRekamanFolderRef.get();
//     if (!folderSnapshot.exists) {
//       console.log(`Folder with ID ${folderId} not found.`);
//       break;
//     }
//     const folderData = folderSnapshot.data();
//     const assignedUsers = folderData?.access?.assignedUsers || [];
//     const userExists = assignedUsers.some(
//       (user: any) => user.id === userObj.id
//     );
//     if (!userExists) {
//       await folderRef.update({
//         "access.assignedUsers": FieldValue.arrayUnion(userObj),
//         "access.assignedUsersId": FieldValue.arrayUnion(userObj.id),
//       });
//       if (searchableRekamanFolderSnapshot.exists) {
//         await searchableRekamanFolderRef.update({
//           "access.assignedUsers": FieldValue.arrayUnion(userObj),
//           "access.assignedUsersId": FieldValue.arrayUnion(userObj.id),
//         });
//       }
//       console.log(`User ${userObj.id} added to folder ${folderId}.`);
//     } else {
//       console.log(
//         `User ${userObj.id} already has access to folder ${folderId}.`
//       );
//     }
//     folderId = folderData?.parentFolderId || null;
//   }
// };
const grantFolderAccess = (workspaceRef_1, companyId_1, userId_1, userName_1, userPhotoURL_1, folderId_1, invitedUsers_1, ...args_1) => __awaiter(void 0, [workspaceRef_1, companyId_1, userId_1, userName_1, userPhotoURL_1, folderId_1, invitedUsers_1, ...args_1], void 0, function* (workspaceRef, companyId, userId, userName, userPhotoURL, folderId, invitedUsers, // Only user IDs are passed
folderType = "recording") {
    var _a;
    const addedUsers = [];
    // Fetch full details for each invited user
    const userDetailsPromises = invitedUsers.map((user) => __awaiter(void 0, void 0, void 0, function* () {
        const userDoc = yield firebase_admin_1.adminDb.collection("companies").doc(companyId).collection("users").doc(user.id).get();
        if (!userDoc.exists) {
            console.log(`User with ID ${user.id} not found.`);
            return null;
        }
        const userData = userDoc.data();
        return {
            id: user.id,
            email: (userData === null || userData === void 0 ? void 0 : userData.email) || "",
            name: (userData === null || userData === void 0 ? void 0 : userData.name) || "",
            photoURL: (userData === null || userData === void 0 ? void 0 : userData.photoURL) || null,
        };
    }));
    const resolvedUserDetails = (yield Promise.all(userDetailsPromises)).filter((user) => user !== null);
    while (folderId) {
        const folderRef = workspaceRef.collection("rekamanFolders").doc(folderId);
        const folderSnapshot = yield folderRef.get();
        if (!folderSnapshot.exists) {
            console.log(`Folder with ID ${folderId} not found.`);
            break;
        }
        const folderData = folderSnapshot.data();
        const assignedUsers = ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.assignedUsers) || [];
        for (const userObj of resolvedUserDetails) {
            const userExists = assignedUsers.some((user) => user.id === userObj.id);
            if (!userExists) {
                // Update Firestore with new user access
                yield folderRef.update({
                    "access.assignedUsers": firestore_1.FieldValue.arrayUnion(userObj),
                    "access.assignedUsersId": firestore_1.FieldValue.arrayUnion(userObj.id),
                });
                console.log(`User ${userObj.id} added to folder ${folderId}.`);
                addedUsers.push({
                    id: userObj.id,
                    email: userObj.email,
                    name: userObj.name,
                    photoURL: (userObj === null || userObj === void 0 ? void 0 : userObj.photoURL) || undefined,
                    isRead: false,
                });
            }
            else {
                console.log(`User ${userObj.id} already has access to folder ${folderId}.`);
            }
        }
        // Send notification if this is the top-level parent folder
        if (!(folderData === null || folderData === void 0 ? void 0 : folderData.parentFolderId) && addedUsers.length > 0) {
            yield (0, notifications_1.sendNotificationFolderAccess)(folderData.folderName, workspaceRef.id, userId, userName, userPhotoURL, folderId, "recording", `You have been granted access to the rekaman folder "${folderData.folderName}".`, addedUsers);
            console.log(`Notification sent for folder ${folderId} as it is the top-level folder.`);
            // Clear addedUsers after sending notification
            addedUsers.length = 0;
        }
        folderId = (folderData === null || folderData === void 0 ? void 0 : folderData.parentFolderId) || null;
    }
});
const editRekamanFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f;
    try {
        const { workspaceId, rekamanFolderId } = req.params;
        const user = req.user;
        if (!workspaceId || !rekamanFolderId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID and Rekaman Folder ID are required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to perform this action.",
            });
        }
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user === null || user === void 0 ? void 0 : user.companyId);
        const companySnapshot = yield companyRef.get();
        if (!companySnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Company not found." });
        }
        const parsedBody = divisiFolder_1.editFolderSchema.parse(req.body);
        const { folderName, description, invitedUsers, revokedUsers, newParentFolderId, } = parsedBody;
        const [workspaceDoc, folderDoc] = yield firebase_admin_1.adminDb.getAll(firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId), firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("rekamanFolders")
            .doc(rekamanFolderId));
        if (!workspaceDoc.exists ||
            ((_a = workspaceDoc.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You do not have permission to edit rekaman folders in this workspace.",
            });
        }
        if (!folderDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Rekaman Folder not found.",
            });
        }
        const folderData = folderDoc.data();
        const isFolderEditor = (_c = (_b = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _b === void 0 ? void 0 : _b.assignedUsers) === null || _c === void 0 ? void 0 : _c.some((u) => u.id === user.id && u.role === "editor");
        // Access permissions: Allow actions if the user is superadmin, admin, or editor within assignedUsers
        const isAdminOrEditor = ["superadmin", "admin"].includes(user === null || user === void 0 ? void 0 : user.role) ||
            ((_e = (_d = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _d === void 0 ? void 0 : _d.assignedUsers) === null || _e === void 0 ? void 0 : _e.some((u) => u.id === user.id && u.role === "editor"));
        const isFolderCreator = (folderData === null || folderData === void 0 ? void 0 : folderData.createdBy) === user.id;
        if (!isAdminOrEditor && !isFolderCreator) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You are not allowed to edit this rekaman folder.",
            });
        }
        // Calculate current folder size
        const currentFolderSize = Buffer.byteLength(folderData === null || folderData === void 0 ? void 0 : folderData.folderName, "utf-8") +
            Buffer.byteLength((folderData === null || folderData === void 0 ? void 0 : folderData.description) || "", "utf-8");
        // Calculate new folder size
        const newFolderSize = Buffer.byteLength(folderName || (folderData === null || folderData === void 0 ? void 0 : folderData.folderName), "utf-8") +
            Buffer.byteLength(description || (folderData === null || folderData === void 0 ? void 0 : folderData.description) || "", "utf-8");
        const sizeDifference = newFolderSize - currentFolderSize;
        // Fetch company data for storage usage validation
        const companyDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .get();
        const companyData = companyDoc.data();
        if (!companyData) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Company not found.",
            });
        }
        const { storageLimit = 0 } = companyData.plan; // Storage limit in MB
        const { usage = {} } = companyData; // Usage field (could be empty initially)
        const currentStorageUsage = usage.storageUsage || 0; // Storage usage in MB
        // Calculate potential new storage usage
        const potentialStorageUsage = currentStorageUsage + sizeDifference / (1024 * 1024); // Convert bytes to MB
        // Check if the potential new usage exceeds the storage limit
        if (potentialStorageUsage > storageLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Allowed: ${storageLimit} MB, Used: ${currentStorageUsage} MB.`,
            });
        }
        const updates = {};
        if (folderName) {
            updates.folderName = folderName;
            updates.folderName_insensitive = folderName.toLowerCase();
        }
        if (description)
            updates.description = description;
        if (isAdminOrEditor && (invitedUsers || revokedUsers)) {
            const updatedAssignedUsers = yield mergeAssignedUsers(((_f = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _f === void 0 ? void 0 : _f.assignedUsers) || [], invitedUsers || [], revokedUsers || [], user.companyId);
            updates["access.assignedUsers"] = updatedAssignedUsers;
            updates["access.assignedUsersId"] = updatedAssignedUsers.map((user) => user.id);
            if (invitedUsers && invitedUsers.length > 0) {
                yield grantFolderAccess(workspaceDoc.ref, user.companyId, user.id, user.name, user.photoURL || undefined, rekamanFolderId, invitedUsers, "recording");
            }
        }
        // Determine the `parentFolderId` logic
        if (newParentFolderId !== undefined) {
            if (newParentFolderId) {
                // If `newParentFolderId` is provided, ensure it exists
                const newParentFolderDoc = yield firebase_admin_1.adminDb
                    .collection("workspaces")
                    .doc(workspaceId)
                    .collection("rekamanFolders")
                    .doc(newParentFolderId)
                    .get();
                if (!newParentFolderDoc.exists) {
                    return res.status(404).json({
                        status: "error",
                        code: 404,
                        message: "New parent folder not found.",
                    });
                }
                updates.parentFolderId = newParentFolderId;
            }
        }
        else if (!folderName && !description && !invitedUsers) {
            // If no relevant fields are provided and newParentFolderId is not given, make it a top-level folder
            updates.parentFolderId = null;
        }
        // Only proceed if there are actual updates to apply
        if (Object.keys(updates).length === 0) {
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "No changes were made.",
            });
        }
        updates.updatedAt = new Date().toISOString();
        updates.updatedBy = user.id;
        yield folderDoc.ref.update(updates);
        const searchRekamanFolderRef = firebase_admin_1.adminDb
            .collection("searchableRekamanFolder")
            .doc(rekamanFolderId);
        const folderDataForSearch = Object.assign(Object.assign(Object.assign({}, folderData), updates), { companyId: user.companyId });
        yield searchRekamanFolderRef.set(folderDataForSearch);
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .update({
            "usage.storageUsage": firestore_1.FieldValue.increment(sizeDifference / (1024 * 1024)), // Increment storage usage
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Rekaman Folder updated successfully.",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.editRekamanFolder = editRekamanFolder;
// const viewAllRekamanFolders = async (
//   req: AuthenticatedRequest,
//   res: Response
// ) => {
//   try {
//     const { workspaceId } = req.params;
//     const user = req.user;
//     if (!workspaceId) {
//       return res.status(400).json({
//         status: "error",
//         code: 400,
//         message: "Workspace ID is required.",
//       });
//     }
//     if (!user || !user.companyId) {
//       return res.status(403).json({
//         status: "error",
//         code: 403,
//         message: "User is not authorized to view folders.",
//       });
//     }
//     const workspaceDoc = await getDoc(doc(db, "workspaces", workspaceId));
//     if (!workspaceDoc.exists()) {
//       return res.status(404).json({
//         status: "error",
//         code: 404,
//         message: "Workspace not found.",
//       });
//     }
//     const workspaceData = workspaceDoc.data();
//     if (workspaceData?.companyId !== user.companyId) {
//       return res.status(403).json({
//         status: "error",
//         code: 403,
//         message:
//           "Access denied. You do not have permission to view rekaman folders in this workspace.",
//       });
//     }
//     const perPage = parseInt(req.query.perPage as string) || 10;
//     const startAfterDocId = (req.query.startAfter as string) || null;
//     const searchTerm = ((req.query.q as string) || "").toLowerCase();
//     const foldersRef = collection(
//       db,
//       "workspaces",
//       workspaceId,
//       "rekamanFolders"
//     );
//     const userData = {
//       email: user.email,
//       id: user.id,
//       name: user.name,
//       photoURL: user.photoURL || null,
//     };
//     let baseQuery = query(
//       foldersRef,
//       and(
//         where("parentFolderId", "==", null),
//         or(
//           where("access.allowedRoles", "array-contains", user.role),
//           where("access.assignedUsers", "array-contains", {
//             ...userData,
//             role: "editor",
//           }),
//           where("access.assignedUsers", "array-contains", {
//             ...userData,
//             role: "viewer",
//           })
//         )
//       ),
//       orderBy(searchTerm ? "folderName_insensitive" : "createdAt"),
//       limit(perPage)
//     );
//     if (startAfterDocId) {
//       const startAfterDoc = await getDoc(
//         doc(db, "workspaces", workspaceId, "rekamanFolders", startAfterDocId)
//       );
//       if (startAfterDoc.exists()) {
//         baseQuery = query(baseQuery, startAfter(startAfterDoc));
//       } else {
//         return res.status(400).json({
//           status: "error",
//           code: 400,
//           message: `Rekaman Folders with startAfterDocId ${startAfterDocId} does not exist.`,
//         });
//       }
//     }
//     const parentFoldersSnapshot = await getDocs(baseQuery);
//     if (parentFoldersSnapshot.empty) {
//       return res.status(200).json({
//         status: "success",
//         code: 200,
//         message: "No rekaman folders found in this workspace.",
//         data: [],
//         lastVisible: null,
//       });
//     }
//     const countAllRekaman = async (folderId: string): Promise<number> => {
//       let totalCount = 0;
//       const rekamanSnapshot = await getDocs(
//         query(
//           collection(db, "rekaman"),
//           and(
//             where("rekamanFolderId", "==", folderId),
//             or(
//               where("access.ownerId", "==", user.id),
//               where("access.allowedRoles", "array-contains", user.role),
//               where("access.invitedUsers", "array-contains", {
//                 ...userData,
//                 role: "editor",
//               }),
//               where("access.invitedUsers", "array-contains", {
//                 ...userData,
//                 role: "viewer",
//               })
//             )
//           )
//         )
//       );
//       totalCount += rekamanSnapshot.size;
//       const subfoldersSnapshot = await getDocs(
//         query(foldersRef, where("parentFolderId", "==", folderId))
//       );
//       for (const subfolderDoc of subfoldersSnapshot.docs) {
//         totalCount += await countAllRekaman(subfolderDoc.id);
//       }
//       return totalCount;
//     };
//     const parentFolders = await Promise.all(
//       parentFoldersSnapshot.docs.map(async (folderDoc) => {
//         const data = folderDoc.data();
//         const rekamanCount = await countAllRekaman(folderDoc.id);
//         // Fetch createdByUser information
//         const createdByUserRef = doc(
//           db,
//           "companies",
//           user?.companyId!,
//           "users",
//           data.createdBy
//         );
//         const createdByUserDoc = await getDoc(createdByUserRef);
//         const createdByUser = createdByUserDoc.exists()
//           ? {
//               name: createdByUserDoc.data().name,
//               email: createdByUserDoc.data().email,
//               photoURL: createdByUserDoc.data().photoURL || null,
//             }
//           : null;
//         return {
//           id: folderDoc.id,
//           ...data,
//           rekamanCount,
//           createdByUser,
//         };
//       })
//     );
//     const lastVisible =
//       parentFoldersSnapshot.docs[parentFoldersSnapshot.docs.length - 1];
//     return res.status(200).json({
//       status: "success",
//       code: 200,
//       message: "Rekaman folders fetched successfully.",
//       data: parentFolders,
//       lastVisible: lastVisible ? lastVisible.id : null,
//     });
//   } catch (error) {
//     console.log(error);
//     return res.status(500).json({
//       status: "error",
//       code: 500,
//       message: "Internal Server Error",
//     });
//   }
// };
const viewAllRekamanFolders = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const user = req.user;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to view folders.",
            });
        }
        const workspaceDoc = yield (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "workspaces", workspaceId));
        if (!workspaceDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied.",
            });
        }
        const foldersRef = (0, firestore_2.collection)(firebase_1.db, "workspaces", workspaceId, "rekamanFolders");
        const perPage = parseInt(req.query.perPage) || 10;
        const startAfterDocId = req.query.startAfter;
        const searchTerm = (req.query.q || "").toLowerCase();
        const page = parseInt(req.query.page) || 1;
        const userData = {
            email: user.email,
            id: user.id,
            name: user.name,
            photoURL: user.photoURL || null,
        };
        let countQuery = (0, firestore_2.query)(foldersRef, (0, firestore_2.and)((0, firestore_2.where)("parentFolderId", "==", null), (0, firestore_2.or)((0, firestore_2.where)("access.allowedRoles", "array-contains", user.role), (0, firestore_2.where)("access.assignedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "editor" })), (0, firestore_2.where)("access.assignedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "viewer" })))));
        let baseQuery = (0, firestore_2.query)(foldersRef, (0, firestore_2.and)((0, firestore_2.where)("parentFolderId", "==", null), (0, firestore_2.or)((0, firestore_2.where)("access.allowedRoles", "array-contains", user.role), (0, firestore_2.where)("access.assignedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "editor" })), (0, firestore_2.where)("access.assignedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "viewer" })))), (0, firestore_2.orderBy)(searchTerm ? "folderName_insensitive" : "createdAt"), (0, firestore_2.limit)(perPage));
        let totalRekamanFolders = 0;
        let folderIdsFromSearch = [];
        if (searchTerm) {
            try {
                const searchParameters = {
                    q: searchTerm,
                    query_by: "folderName_insensitive",
                    filter_by: `workspaceId:${workspaceId} && companyId:${user.companyId} && (
             access.allowedRoles:=${user.role} || access.assignedUsersId:=${user.id} 
          )`
                        .replace(/\s+/g, " ")
                        .trim(),
                    per_page: perPage,
                    page,
                };
                const searchResults = yield typesense_1.default
                    .collections("searchableRekamanFolder")
                    .documents()
                    .search(searchParameters);
                folderIdsFromSearch = searchResults === null || searchResults === void 0 ? void 0 : searchResults.hits.map((hit) => { var _a; return (_a = hit.document) === null || _a === void 0 ? void 0 : _a.id; });
                totalRekamanFolders = searchResults.found;
            }
            catch (error) {
                console.log(error);
            }
        }
        if (searchTerm && folderIdsFromSearch.length === 0) {
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "No rekaman folders found matching the search criteria.",
                data: [],
                totalDivisiFolders: 0,
                lastVisible: null,
            });
        }
        if (folderIdsFromSearch.length > 0) {
            baseQuery = (0, firestore_2.query)(foldersRef, (0, firestore_2.where)("id", "in", folderIdsFromSearch));
            countQuery = (0, firestore_2.query)(foldersRef, (0, firestore_2.where)("id", "in", folderIdsFromSearch));
        }
        if (!searchTerm) {
            const totalFoldersSnapshot = yield (0, firestore_2.getDocs)(countQuery);
            totalRekamanFolders = totalFoldersSnapshot.size;
        }
        if (!searchTerm && startAfterDocId) {
            const startAfterDoc = yield (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "workspaces", workspaceId, "rekamanFolders", startAfterDocId));
            if (startAfterDoc.exists()) {
                baseQuery = (0, firestore_2.query)(baseQuery, (0, firestore_2.startAfter)(startAfterDoc));
            }
            else {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: `Document with startAfterDocId ${startAfterDocId} does not exist.`,
                });
            }
        }
        const parentFoldersSnapshot = yield (0, firestore_2.getDocs)(baseQuery);
        if (parentFoldersSnapshot.empty) {
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "No rekaman folders found.",
                data: [],
                totalRekamanFolders: 0,
                lastVisible: null,
            });
        }
        const fetchDocumentCounts = (folderId) => __awaiter(void 0, void 0, void 0, function* () {
            const rekamanSnapshot = yield (0, firestore_2.getDocs)((0, firestore_2.query)((0, firestore_2.collection)(firebase_1.db, "rekaman"), (0, firestore_2.and)((0, firestore_2.where)("rekamanFolderId", "==", folderId), (0, firestore_2.or)((0, firestore_2.where)("access.ownerId", "==", user.id), (0, firestore_2.where)("access.allowedRoles", "array-contains", user.role), (0, firestore_2.where)("access.invitedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "editor" })), (0, firestore_2.where)("access.invitedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "viewer" }))))));
            return rekamanSnapshot.size;
        });
        const fetchSubfoldersWithCounts = (parentFolderId) => __awaiter(void 0, void 0, void 0, function* () {
            const subfoldersSnapshot = yield (0, firestore_2.getDocs)((0, firestore_2.query)(foldersRef, (0, firestore_2.where)("parentFolderId", "==", parentFolderId)));
            const subfolders = yield Promise.all(subfoldersSnapshot.docs.map((subfolderDoc) => __awaiter(void 0, void 0, void 0, function* () {
                const data = subfolderDoc.data();
                const documentCount = yield fetchDocumentCounts(subfolderDoc.id);
                const nestedSubfolders = yield fetchSubfoldersWithCounts(subfolderDoc.id);
                const totalDocumentCount = documentCount +
                    nestedSubfolders.reduce((sum, sub) => sum + sub.documentCount, 0);
                return Object.assign(Object.assign({ id: subfolderDoc.id }, data), { documentCount: totalDocumentCount, subfolders: nestedSubfolders });
            })));
            return subfolders;
        });
        const parentFolders = yield Promise.all(parentFoldersSnapshot.docs.map((folderDoc) => __awaiter(void 0, void 0, void 0, function* () {
            const data = folderDoc.data();
            const documentCount = yield fetchDocumentCounts(folderDoc.id);
            const subfolders = yield fetchSubfoldersWithCounts(folderDoc.id);
            const totalDocumentCount = documentCount +
                subfolders.reduce((sum, sub) => sum + sub.documentCount, 0);
            const createdByUserDoc = yield (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "companies", user.companyId, "users", data.createdBy));
            const createdByUser = createdByUserDoc.exists()
                ? createdByUserDoc.data()
                : { name: "Unknown User", email: "No Email", photoURL: null };
            return Object.assign(Object.assign({ id: folderDoc.id }, data), { createdByUser, documentCount: totalDocumentCount, subfolders });
        })));
        const lastVisible = parentFoldersSnapshot.docs[parentFoldersSnapshot.docs.length - 1];
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Rekaman folders fetched successfully.",
            data: parentFolders,
            totalRekamanFolders,
            lastVisible: lastVisible ? lastVisible.id : null,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewAllRekamanFolders = viewAllRekamanFolders;
const viewRekamanFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { workspaceId, rekamanFolderId } = req.params;
        const user = req.user;
        if (!workspaceId || !rekamanFolderId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID and Folder ID are required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to perform this action.",
            });
        }
        // Check if the workspace exists and belongs to the user's company
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You do not have permission to view folders in this workspace.",
            });
        }
        // Fetch the main rekaman folder
        const folderDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("rekamanFolders")
            .doc(rekamanFolderId)
            .get();
        if (!folderDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Rekaman Folder not found.",
            });
        }
        const folderData = folderDoc.data();
        // Check access permissions based on roles
        const isAdminOrSuperadmin = (user === null || user === void 0 ? void 0 : user.role) === "superadmin" || (user === null || user === void 0 ? void 0 : user.role) === "admin";
        const isUserAssigned = (_b = (_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.assignedUsers) === null || _b === void 0 ? void 0 : _b.some((assignedUser) => assignedUser.id === user.id);
        if (!isAdminOrSuperadmin && !isUserAssigned) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You are not allowed to view this folder.",
            });
        }
        // Fetch subfolders once
        const subfoldersSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("rekamanFolders")
            .where("parentFolderId", "==", rekamanFolderId)
            .get();
        // Get the parent folder name once
        const parentFolderName = (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || null;
        // Map the subfolders and reuse the parentFolderName
        const subfolders = subfoldersSnapshot.docs.map((doc) => (Object.assign({ id: doc.id, parentFolderName }, doc.data())));
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Folder and subfolders fetched successfully.",
            data: {
                folder: Object.assign({ id: folderDoc.id }, folderData),
                subfolders,
            },
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewRekamanFolder = viewRekamanFolder;
const deleteRekamanFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        req.body = { rekamanFolderId: req.params.rekamanFolderId };
        return yield (0, trash_1.moveRekamanToTrash)(req, res);
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteRekamanFolder = deleteRekamanFolder;
