"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteRekaman = exports.viewRekaman = exports.viewAllRekamanInFolder = exports.viewAllRekaman = exports.editRekaman = exports.createRekaman = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const zod_1 = require("zod");
const rekaman_1 = require("../../../../types/rekaman");
const trash_1 = require("./trash");
const notifications_1 = require("./notifications");
const firestore_1 = require("firebase-admin/firestore");
const crypto_1 = __importDefault(require("crypto"));
const createRekaman = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { workspaceId } = req.params;
        const parsedBody = rekaman_1.createRekamanSchema.parse(req.body);
        const { isManual, title, content, extension, rekamanFolderId } = parsedBody;
        const user = req.user;
        // Step 1: Validate workspace ownership
        const workspaceRef = firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId);
        const workspaceSnapshot = yield workspaceRef.get();
        if (!workspaceSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceSnapshot.data();
        if (workspaceData.companyId !== user.companyId) {
            return res
                .status(403)
                .json({ status: "error", code: 403, message: "Access denied." });
        }
        // Step 2: Check folder existence and user access
        const folderRef = workspaceRef
            .collection("rekamanFolders")
            .doc(rekamanFolderId);
        const folderSnapshot = yield folderRef.get();
        if (!folderSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Rekaman Folder not found.",
            });
        }
        const folderData = folderSnapshot.data();
        const hasAccess = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles.includes(user.role)) ||
            (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers.some((assignedUser) => assignedUser.id === user.id)) ||
            ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.ownerId) === user.id;
        if (!hasAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied to this folder.",
            });
        }
        // Step 4: Fetch user details from the company’s users subcollection
        const userRef = firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users")
            .doc(user.id);
        const userSnapshot = yield userRef.get();
        const userData = userSnapshot.data();
        const formatDateIndonesian = (date) => {
            return new Intl.DateTimeFormat("id-ID", {
                day: "2-digit",
                month: "long",
                year: "numeric",
                hour: "2-digit",
                minute: "2-digit",
                hour12: true,
            }).format(date);
        };
        const generateDocumentKey = () => {
            return crypto_1.default.randomBytes(16).toString("hex"); // Example 16-byte random key
        };
        let newKey = generateDocumentKey();
        let documentData = {
            title,
            rekamanFolderId,
            folderName: (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || "Unknown Folder",
            workspaceId,
            createdAt: new Date().toISOString(),
            createdBy: user.id,
            createdByName: (userData === null || userData === void 0 ? void 0 : userData.name) || "Unknown User",
            createdByEmail: (userData === null || userData === void 0 ? void 0 : userData.email) || "Unknown Email",
            createdByPhotoURL: (userData === null || userData === void 0 ? void 0 : userData.photoURL) || null,
            access: {
                ownerId: user.id,
                allowedRoles: ["admin", "superadmin"],
                invitedUsers: [],
            },
            currentVersion: 1,
        };
        let filePath;
        let newDocumentSize = 0; // Initialize size in MB
        let fileExtension = ""; // Store the file extension
        if (isManual) {
            const contentSize = Buffer.byteLength(content || "", "utf-8");
            const titleSize = Buffer.byteLength(title, "utf-8");
            const firestoreOverhead = 1024; // Approximate Firestore document overhead in bytes
            // Total size in MB, including title, content, and Firestore overhead
            newDocumentSize =
                (contentSize + titleSize + firestoreOverhead) / (1024 * 1024);
            documentData.content = content;
            fileExtension = extension || "docx";
            documentData.fileExtension = fileExtension;
        }
        else {
            // File upload handling
            const docs = req.file; // Assuming file comes from 'multipart/form-data'
            if (!docs) {
                return res
                    .status(400)
                    .json({ status: "error", code: 400, message: "No file uploaded." });
            }
            fileExtension = docs.originalname.split(".").pop() || "unknown";
            const sanitizedTitle = title
                .replace(/[^a-zA-Z0-9-_ ]/g, "")
                .replace(/ /g, "_");
            const newFileName = `${sanitizedTitle}.${fileExtension}`;
            filePath = `${user.companyId}/workspaces/${workspaceId}/rekamanFolders/${rekamanFolderId}/rekaman/${newFileName}`;
            const storageFile = firebase_admin_1.bucket.file(filePath);
            yield storageFile.save(docs.buffer, {
                metadata: { contentType: docs.mimetype },
            });
            documentData.storagePath = filePath;
            const historyEntry = {
                created: formatDateIndonesian(new Date()),
                key: newKey,
                user: {
                    id: user.id,
                    name: user.name || "Unknown User",
                },
                historyData: {
                    fileType: fileExtension,
                    key: newKey,
                    url: filePath,
                },
                version: 1,
            };
            documentData.history = [historyEntry];
            documentData.key = newKey;
            documentData.fileExtension = fileExtension;
            // Calculate the uploaded file size in MB
            newDocumentSize = docs.size / (1024 * 1024);
        }
        // Calculate document size and update storage usage
        const companyDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .get();
        const companyData = companyDoc.data();
        if (!companyData) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Company not found.",
            });
        }
        const { storageLimit = 0 } = companyData.plan; // Storage limit in MB
        const { usage = {} } = companyData; // Usage field (could be empty initially)
        const currentStorageUsage = usage.storageUsage || 0; // Storage usage in MB
        // Check if the potential new usage exceeds the storage limit
        const potentialStorageUsage = currentStorageUsage + newDocumentSize;
        if (potentialStorageUsage > storageLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Allowed: ${storageLimit} MB, Used: ${currentStorageUsage} MB.`,
            });
        }
        // Save document to Firestore
        const docRef = firebase_admin_1.adminDb.collection("rekaman").doc();
        yield docRef.set(Object.assign(Object.assign({}, documentData), { id: docRef.id }));
        // Update the company's storage usage
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .update({
            "usage.storageUsage": firestore_1.FieldValue.increment(newDocumentSize),
        });
        return res.status(201).json({
            status: "success",
            message: "Rekaman created successfully.",
            data: documentData,
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.createRekaman = createRekaman;
const grantFolderAccessToUser = (workspaceRef_1, folderId_1, userId_1, email_1, name_1, role_1, photoURL_1, ...args_1) => __awaiter(void 0, [workspaceRef_1, folderId_1, userId_1, email_1, name_1, role_1, photoURL_1, ...args_1], void 0, function* (workspaceRef, folderId, userId, email, name, role, photoURL, folderType = "folders") {
    var _a;
    const userObj = {
        id: String(userId),
        email,
        name,
        role,
        photoURL,
    };
    while (folderId) {
        const folderRef = workspaceRef.collection("rekamanFolders").doc(folderId);
        const folderSnapshot = yield folderRef.get();
        if (!folderSnapshot.exists) {
            console.log(`Folder with ID ${folderId} not found.`);
            break;
        }
        const folderData = folderSnapshot.data();
        let assignedUsers = ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.assignedUsers) || [];
        // Check if user already exists in assignedUsers based on userId
        const userExists = assignedUsers.some((user) => user.userId === userObj.id);
        if (!userExists) {
            // Add the user object to access.assignedUsers and update the field
            yield folderRef.update({
                "access.assignedUsers": firestore_1.FieldValue.arrayUnion(userObj),
                "access.assignedUsersId": firestore_1.FieldValue.arrayUnion(userObj.id),
            });
            console.log(`User ${userObj.id} added to folder ${folderId}.`);
        }
        else {
            console.log(`User ${userObj.id} already exists in folder ${folderId}.`);
        }
        // Move to the next parent folder in the hierarchy
        folderId = folderData.parentFolderId || null;
    }
});
const editRekaman = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f, _g;
    try {
        const { rekamanId, workspaceId, rekamanFolderId } = req.params;
        const parsedBody = rekaman_1.editRekamanSchema.parse(req.body);
        const user = req.user;
        console.log(req.body.invitedUsers);
        const workspaceRef = firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId);
        const rekamanRef = firebase_admin_1.adminDb.collection("rekaman").doc(rekamanId);
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user.companyId);
        const usersRef = companyRef.collection("users");
        const requests = [workspaceRef.get(), rekamanRef.get(), companyRef.get()];
        if (parsedBody.newRekamanFolderId) {
            const newFolderRef = workspaceRef
                .collection("rekamanFolders")
                .doc(parsedBody.newRekamanFolderId);
            requests.push(newFolderRef.get());
        }
        const [workspaceSnapshot, rekamanSnapshot, companySnapshot] = yield Promise.all(requests);
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Workspace access denied.",
            });
        }
        if (!rekamanSnapshot.exists ||
            ((_b = rekamanSnapshot.data()) === null || _b === void 0 ? void 0 : _b.rekamanFolderId) !== rekamanFolderId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Rekaman not found.",
            });
        }
        const rekamanData = rekamanSnapshot.data();
        const isOwnerOrAdmin = (rekamanData === null || rekamanData === void 0 ? void 0 : rekamanData.access.ownerId) === user.id ||
            ["admin", "superadmin"].includes(user.role);
        const isEditorInvitedUser = (_c = rekamanData === null || rekamanData === void 0 ? void 0 : rekamanData.access.invitedUsers) === null || _c === void 0 ? void 0 : _c.some((invitedUser) => invitedUser.id === user.id && invitedUser.role === "editor");
        if (!isOwnerOrAdmin && !isEditorInvitedUser) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to edit this rekaman.",
            });
        }
        if (!companySnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Company not found.",
            });
        }
        const companyData = companySnapshot.data();
        const existingInvitedUsers = (rekamanData === null || rekamanData === void 0 ? void 0 : rekamanData.access.invitedUsers) || [];
        const updatedInvitedUsers = [...existingInvitedUsers];
        const invitedUsers = parsedBody.invitedUsers || [];
        const fetchedUsers = [];
        for (const invitedUser of invitedUsers) {
            const companyUserSnapshot = yield usersRef.doc(invitedUser.id).get();
            if (!companyUserSnapshot.exists) {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: `User ID ${invitedUser.id} not found.`,
                });
            }
            const companyUser = companyUserSnapshot.data();
            const updatedUser = {
                id: companyUser.id,
                email: companyUser.email,
                name: companyUser.name,
                role: invitedUser.role,
                photoURL: (companyUser === null || companyUser === void 0 ? void 0 : companyUser.photoURL) || null,
            };
            const existingUserIndex = updatedInvitedUsers.findIndex((user) => user.id === invitedUser.id);
            if (existingUserIndex !== -1) {
                updatedInvitedUsers[existingUserIndex] = updatedUser;
            }
            else {
                updatedInvitedUsers.push(updatedUser);
            }
            fetchedUsers.push(updatedUser);
            yield grantFolderAccessToUser(workspaceRef, rekamanFolderId, invitedUser.id, companyUser.email, companyUser.name, invitedUser.role, companyUser.photoURL || null);
        }
        if (invitedUsers.length > 0) {
            const documentType = "record";
            const formattedInvitedUsers = fetchedUsers.map((user) => ({
                id: user.id,
                email: user.email || "Unknown Email",
                name: user.name || "Unknown User",
                photoURL: user.photoURL || null,
                isRead: false,
            }));
            yield Promise.all(invitedUsers.map(() => (0, notifications_1.sendNotificationRekamanAccess)(rekamanId, rekamanData === null || rekamanData === void 0 ? void 0 : rekamanData.title, rekamanData === null || rekamanData === void 0 ? void 0 : rekamanData.fileExtension, user.id, user.name, (user === null || user === void 0 ? void 0 : user.photoURL) || undefined, user.companyId, workspaceId, rekamanFolderId, documentType, `Rekaman "${rekamanData === null || rekamanData === void 0 ? void 0 : rekamanData.title}" has been shared with you by ${user.name}.`, formattedInvitedUsers)));
        }
        const { invitedUsers: invitedUsersFromBody, revokedUsers: revokedUsersFromBody } = parsedBody, restOfBody = __rest(parsedBody, ["invitedUsers", "revokedUsers"]);
        const revokedUsers = revokedUsersFromBody || [];
        const revokedUserIds = revokedUsers.map((user) => user.id);
        const finalInvitedUsers = updatedInvitedUsers.filter((user) => !revokedUserIds.includes(user.id));
        const updatedRekamanData = Object.assign(Object.assign({}, restOfBody), { rekamanFolderId: parsedBody.newRekamanFolderId || rekamanFolderId, updatedAt: new Date().toISOString(), updatedBy: user.id, "access.invitedUsers": finalInvitedUsers });
        let newContentSize = 0;
        let oldContentSize = 0;
        if (parsedBody.content) {
            newContentSize = Buffer.byteLength(parsedBody.content, "utf-8");
        }
        if (rekamanData === null || rekamanData === void 0 ? void 0 : rekamanData.content) {
            oldContentSize = Buffer.byteLength(rekamanData.content, "utf-8");
        }
        const sizeDifference = (newContentSize - oldContentSize) / (1024 * 1024);
        if (sizeDifference > 0 &&
            (((_d = companyData === null || companyData === void 0 ? void 0 : companyData.usage) === null || _d === void 0 ? void 0 : _d.storageUsage) || 0) + sizeDifference >
                (((_e = companyData === null || companyData === void 0 ? void 0 : companyData.plan) === null || _e === void 0 ? void 0 : _e.storageLimit) || 0)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Your plan allows ${(_f = companyData === null || companyData === void 0 ? void 0 : companyData.plan) === null || _f === void 0 ? void 0 : _f.storageLimit} MB, but the current usage is ${(_g = companyData === null || companyData === void 0 ? void 0 : companyData.usage) === null || _g === void 0 ? void 0 : _g.storageUsage} MB.`,
            });
        }
        yield firebase_admin_1.adminDb.runTransaction((transaction) => __awaiter(void 0, void 0, void 0, function* () {
            transaction.update(rekamanRef, updatedRekamanData);
            if (sizeDifference !== 0) {
                transaction.update(companyRef, {
                    "usage.storageUsage": firestore_1.FieldValue.increment(sizeDifference),
                });
            }
        }));
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Rekaman updated successfully.",
        });
    }
    catch (error) {
        console.log(error);
        if (error instanceof zod_1.z.ZodError) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: error.errors[0],
            });
        }
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.editRekaman = editRekaman;
const viewRekaman = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { rekamanId, workspaceId, rekamanFolderId } = req.params;
        const user = req.user;
        // Step 0: Validate rekamanid, workspaceId, and rekamanFolderId
        if (!rekamanId || !workspaceId || !rekamanFolderId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Invalid or missing rekamanid, workspaceId, or rekamanFolderId.",
            });
        }
        // Step 1: Fetch workspace, folder, and document data in parallel
        const [workspaceSnapshot, folderSnapshot, rekamanSnapshot] = yield Promise.all([
            firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId).get(),
            firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("rekamanFolders")
                .doc(rekamanFolderId)
                .get(),
            firebase_admin_1.adminDb.collection("rekaman").doc(rekamanId).get(),
        ]);
        // Check if workspace exists and belongs to the user's company
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found or access denied.",
            });
        }
        // Check if folder exists within the workspace and if the user has access
        if (!folderSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Rekaman Folder not found.",
            });
        }
        const folderData = folderSnapshot.data();
        const hasFolderAccess = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles.includes(user.role)) ||
            (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers.includes(user.id));
        if (!hasFolderAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access to this folder is denied.",
            });
        }
        // Check if document exists and if the user has access to it
        if (!rekamanSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Rekaman not found." });
        }
        const documentData = rekamanSnapshot.data();
        const hasDocumentAccess = (documentData === null || documentData === void 0 ? void 0 : documentData.access.allowedRoles.includes(user.role)) ||
            (documentData === null || documentData === void 0 ? void 0 : documentData.access.invitedUsers.includes(user.id));
        if (!hasDocumentAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to view this rekaman.",
            });
        }
        // Shallow copy of document data
        const rekamanCopy = Object.assign({}, documentData);
        // Remove rekamanFolderId and categoryId from document copy
        delete rekamanCopy.rekamanFolderId;
        delete rekamanCopy.categoryId;
        delete rekamanCopy.folderName;
        // Return document data along with folder and category objects
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Rekaman retrieved successfully.",
            data: Object.assign(Object.assign({}, rekamanCopy), { folder: {
                    id: rekamanFolderId,
                    folderName: (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || "Unknown Folder",
                } }),
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.viewRekaman = viewRekaman;
const viewAllRekaman = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const user = req.user;
        const { workspaceId } = req.params;
        // Fetch the workspace to check company access
        const workspaceSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access to this workspace is denied.",
            });
        }
        // Fetch all documents in the specified workspace
        const rekamanSnapshot = yield firebase_admin_1.adminDb
            .collection("rekaman")
            .where("workspaceId", "==", workspaceId)
            .get();
        // Filter documents based on access control
        const accessibleDocuments = rekamanSnapshot.docs
            .map((doc) => doc.data())
            .filter((document) => {
            var _a;
            const { access } = document;
            const isOwner = (access === null || access === void 0 ? void 0 : access.ownerId) === (user === null || user === void 0 ? void 0 : user.id);
            const isAdminOrSuperAdmin = ["admin", "superadmin"].includes(user === null || user === void 0 ? void 0 : user.role);
            const isInvited = (_a = access === null || access === void 0 ? void 0 : access.invitedUsers) === null || _a === void 0 ? void 0 : _a.includes(user === null || user === void 0 ? void 0 : user.id);
            return isOwner || isAdminOrSuperAdmin || isInvited;
        });
        // Fetch folder and category data
        const folderIds = Array.from(new Set(accessibleDocuments.map((doc) => doc.rekamanFolderId)));
        const folderSnapshots = yield Promise.all(folderIds.map((id) => firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("rekamanFolders")
            .doc(id)
            .get()));
        const folders = folderSnapshots.reduce((acc, snap) => {
            if (snap.exists) {
                acc[snap.id] = snap.data();
            }
            return acc;
        }, {});
        // Fetch user data from the company's users subcollection
        const userIds = Array.from(new Set(accessibleDocuments.map((doc) => doc.createdBy)));
        const userSnapshots = yield Promise.all(userIds.map((id) => firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId) // User's company ID
            .collection("users")
            .doc(id)
            .get()));
        const users = userSnapshots.reduce((acc, snap) => {
            var _a, _b, _c;
            if (snap.exists) {
                acc[snap.id] = {
                    name: (_a = snap.data()) === null || _a === void 0 ? void 0 : _a.name,
                    email: (_b = snap.data()) === null || _b === void 0 ? void 0 : _b.email,
                    photoURL: (_c = snap.data()) === null || _c === void 0 ? void 0 : _c.photoURL, // Include photoURL
                };
            }
            return acc;
        }, {});
        // Attach folderName, categoryName, and createdBy (username, email, and photoURL) to each document
        const documentsWithFoldersCategoriesAndCreator = accessibleDocuments.map((doc) => {
            var _a, _b, _c, _d;
            const rekamanCopy = Object.assign({}, doc);
            delete rekamanCopy.rekamanFolderId;
            delete rekamanCopy.folderName;
            return Object.assign(Object.assign({}, rekamanCopy), { folder: {
                    id: doc.rekamanFolderId,
                    folderName: ((_a = folders[doc.rekamanFolderId]) === null || _a === void 0 ? void 0 : _a.folderName) || "Unknown Folder",
                }, createdBy: {
                    id: doc.createdBy,
                    name: ((_b = users[doc.createdBy]) === null || _b === void 0 ? void 0 : _b.name) || "Unknown User",
                    email: ((_c = users[doc.createdBy]) === null || _c === void 0 ? void 0 : _c.email) || "Unknown Email",
                    photoURL: ((_d = users[doc.createdBy]) === null || _d === void 0 ? void 0 : _d.photoURL) || null, // Add photoURL
                } });
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Rekaman fetched successfully.",
            data: documentsWithFoldersCategoriesAndCreator,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewAllRekaman = viewAllRekaman;
const viewAllRekamanInFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c;
    try {
        const { workspaceId, rekamanFolderId } = req.params;
        const user = req.user;
        const [workspaceSnapshot, folderSnapshot] = yield Promise.all([
            firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId).get(),
            firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("rekamanFolders")
                .doc(rekamanFolderId)
                .get(),
        ]);
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found or access denied.",
            });
        }
        if (!folderSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Folder not found." });
        }
        const folderData = folderSnapshot.data();
        const isUserAssigned = (_c = (_b = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _b === void 0 ? void 0 : _b.assignedUsers) === null || _c === void 0 ? void 0 : _c.some((assignedUser) => assignedUser.id === user.id);
        const hasFolderAccess = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles.includes(user.role)) ||
            (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers.some((assignedUser) => assignedUser.id === user.id));
        if (!hasFolderAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access to this folder is denied.",
            });
        }
        const documentsSnapshot = yield firebase_admin_1.adminDb
            .collection("rekaman")
            .where("workspaceId", "==", workspaceId)
            .where("rekamanFolderId", "==", rekamanFolderId)
            .get();
        const accessibleDocuments = documentsSnapshot.docs
            .map((doc) => doc.data())
            .filter((document) => {
            var _a;
            const { access } = document;
            const isOwner = (access === null || access === void 0 ? void 0 : access.ownerId) === (user === null || user === void 0 ? void 0 : user.id);
            const isAdminOrSuperAdmin = ["admin", "superadmin"].includes(user === null || user === void 0 ? void 0 : user.role);
            const isInvited = (_a = access === null || access === void 0 ? void 0 : access.invitedUsers) === null || _a === void 0 ? void 0 : _a.includes(user === null || user === void 0 ? void 0 : user.id);
            return isOwner || isAdminOrSuperAdmin || isInvited;
        });
        // Fetch user data from the company's users subcollection
        const userIds = Array.from(new Set(accessibleDocuments.map((doc) => doc.createdBy)));
        const userSnapshots = yield Promise.all(userIds.map((id) => firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId) // User's company ID
            .collection("users")
            .doc(id)
            .get()));
        const users = userSnapshots.reduce((acc, snap) => {
            var _a, _b, _c;
            if (snap.exists) {
                acc[snap.id] = {
                    name: (_a = snap.data()) === null || _a === void 0 ? void 0 : _a.name,
                    email: (_b = snap.data()) === null || _b === void 0 ? void 0 : _b.email,
                    photoURL: (_c = snap.data()) === null || _c === void 0 ? void 0 : _c.photoURL, // Include photoURL
                };
            }
            return acc;
        }, {});
        const documentsWithFoldersCategoriesAndCreator = accessibleDocuments.map((doc) => {
            var _a, _b, _c;
            const documentCopy = Object.assign({}, doc);
            delete documentCopy.rekamanFolderId;
            delete documentCopy.categoryId;
            delete documentCopy.folderName;
            return Object.assign(Object.assign({}, documentCopy), { folder: {
                    id: doc.rekamanFolderId,
                    folderName: (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || "Unknown Folder",
                }, createdBy: {
                    id: doc.createdBy,
                    name: ((_a = users[doc.createdBy]) === null || _a === void 0 ? void 0 : _a.name) || "Unknown User",
                    email: ((_b = users[doc.createdBy]) === null || _b === void 0 ? void 0 : _b.email) || "Unknown Email",
                    photoURL: ((_c = users[doc.createdBy]) === null || _c === void 0 ? void 0 : _c.photoURL) || null, // Add photoURL
                } });
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Documents fetched successfully.",
            data: documentsWithFoldersCategoriesAndCreator,
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.viewAllRekamanInFolder = viewAllRekamanInFolder;
const deleteRekaman = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, rekamanFolderId, rekamanId } = req.params;
        const rekamanRef = firebase_admin_1.adminDb.collection("rekaman").doc(rekamanId);
        const rekamanSnapshot = yield rekamanRef.get();
        const rekamanData = rekamanSnapshot.data();
        if (!rekamanSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document not found.",
            });
        }
        // Modify the req.body and req.params to fit what moveToTrash expects
        req.body = { rekamanFolderId, rekamanId };
        req.params = { workspaceId };
        // Call moveToTrash directly
        yield (0, trash_1.moveRekamanToTrash)(req, res);
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteRekaman = deleteRekaman;
