"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.restoreRekamanOffice = exports.trackRekamanOffice = exports.trackDocumentApprovalOffice = exports.viewDocumentApprovalOffice = exports.viewRekamanOffice = exports.createNewRekamanFileFromTemplate = exports.createNewFileFromTemplate = exports.convertDocumentOfficePDF = exports.restoreDocumentOffice = exports.trackDocumentOffice = exports.viewDocumentOffice = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const crypto_1 = __importDefault(require("crypto"));
const path_1 = __importDefault(require("path"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const firestore_1 = require("firebase-admin/firestore");
const onlyoffice_1 = require("../../../../types/onlyoffice");
const axios_1 = __importDefault(require("axios"));
const generateDocumentKey = () => {
    return crypto_1.default.randomBytes(16).toString("hex"); // Example 16-byte random key
};
const formatDateIndonesian = (date) => {
    return new Intl.DateTimeFormat("id-ID", {
        day: "2-digit",
        month: "long",
        year: "numeric",
        hour: "2-digit",
        minute: "2-digit",
        hour12: true,
    }).format(date);
};
const determineDocumentType = (fileType) => {
    const wordExtensions = [
        "doc",
        "docm",
        "docx",
        "dot",
        "dotm",
        "dotx",
        "epub",
        "fb2",
        "fodt",
        "htm",
        "html",
        "mht",
        "mhtml",
        "odt",
        "ott",
        "rtf",
        "stw",
        "sxw",
        "txt",
        "wps",
        "wpt",
        "xml",
    ];
    const cellExtensions = [
        "csv",
        "et",
        "ett",
        "fods",
        "ods",
        "ots",
        "sxc",
        "xls",
        "xlsb",
        "xlsm",
        "xlsx",
        "xlt",
        "xltm",
        "xltx",
        "xml",
    ];
    const slideExtensions = [
        "dps",
        "dpt",
        "fodp",
        "odp",
        "otp",
        "pot",
        "potm",
        "potx",
        "pps",
        "ppsm",
        "ppsx",
        "ppt",
        "pptm",
        "pptx",
        "sxi",
    ];
    const pdfExtensions = ["djvu", "docxf", "oform", "oxps", "pdf", "xps"];
    if (wordExtensions.includes(fileType))
        return "word";
    if (cellExtensions.includes(fileType))
        return "cell";
    if (slideExtensions.includes(fileType))
        return "slide";
    if (pdfExtensions.includes(fileType))
        return "pdf";
    return "unknown"; // Default if no match is found
};
const viewDocumentOffice = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { documentId, workspaceId, divisiFolderId } = req.params;
        const user = req.user;
        // Fetch workspace, folder, and document data in parallel
        const [workspaceSnapshot, folderSnapshot, documentSnapshot] = yield Promise.all([
            firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId).get(),
            firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("folders")
                .doc(divisiFolderId)
                .get(),
            firebase_admin_1.adminDb.collection("documents").doc(documentId).get(),
        ]);
        // Validate workspace
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found or access denied.",
            });
        }
        // Validate folder access
        if (!folderSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Folder not found." });
        }
        const folderData = folderSnapshot.data();
        const folderAllowedRoles = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles) || [];
        const folderAssignedUsers = (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers) || [];
        const folderRequiresApproval = folderData === null || folderData === void 0 ? void 0 : folderData.access.requiresApproval;
        const folderOwner = (folderData === null || folderData === void 0 ? void 0 : folderData.access.ownerId) == user.id;
        const hasFolderAccess = folderAllowedRoles.includes(user.role) ||
            folderAssignedUsers.some((assignedUser) => assignedUser.id === user.id);
        const isAdminOrSuperadmin = user.role === "superadmin" || user.role === "admin";
        if (!(hasFolderAccess ||
            folderRequiresApproval ||
            isAdminOrSuperadmin ||
            folderOwner)) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access to this folder is denied.",
            });
        }
        // Validate document access
        if (!documentSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Document not found." });
        }
        const documentData = documentSnapshot.data();
        const documentAllowedRoles = (documentData === null || documentData === void 0 ? void 0 : documentData.access.allowedRoles) || [];
        const documentInvitedUsers = (documentData === null || documentData === void 0 ? void 0 : documentData.access.invitedUsers) || [];
        const invitedUser = documentInvitedUsers.find((invited) => invited.id === user.id);
        const ownerDocument = (documentData === null || documentData === void 0 ? void 0 : documentData.access.ownerId) == user.id;
        const isViewerOnly = (invitedUser === null || invitedUser === void 0 ? void 0 : invitedUser.role) === "viewer";
        const hasDocumentAccess = documentAllowedRoles.includes(user.role) || invitedUser || ownerDocument;
        if (!hasDocumentAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to view this document.",
            });
        }
        // Ensure the document has a unique key
        let documentKey = documentData === null || documentData === void 0 ? void 0 : documentData.key;
        if (!documentKey) {
            documentKey = generateDocumentKey();
            yield firebase_admin_1.adminDb
                .collection("documents")
                .doc(documentId)
                .update({ key: documentKey });
        }
        // Generate a signed URL for the document
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        const storageFile = firebase_admin_1.bucket.file(filePath);
        const [url] = yield storageFile.getSignedUrl({
            action: "read",
            expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
        });
        // Determine file type
        const extension = path_1.default.extname(filePath).toLowerCase().replace(".", "");
        const fileType = extension || "unknown";
        const documentType = determineDocumentType(fileType);
        // Prepare OnlyOffice configuration
        const onlyOfficeConfig = {
            document: {
                fileType,
                info: {
                    favorite: false,
                    folder: (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || "Unknown Folder",
                    owner: ((_b = documentData === null || documentData === void 0 ? void 0 : documentData.access) === null || _b === void 0 ? void 0 : _b.ownerName) || "Unknown",
                    uploaded: (documentData === null || documentData === void 0 ? void 0 : documentData.createdAt) || "Unknown",
                },
                key: documentKey,
                title: (documentData === null || documentData === void 0 ? void 0 : documentData.title) || "Untitled Document",
                url,
                permissions: {
                    edit: !isViewerOnly,
                    download: !isViewerOnly,
                    print: !isViewerOnly,
                    chat: true,
                    comment: true,
                    copy: !isViewerOnly,
                    fillForms: true,
                    review: true,
                    modifyFilter: false,
                    modifyContentControl: false,
                    protect: false,
                },
            },
            documentType,
            editorConfig: {
                callbackUrl: `https://dmstestapi.mitraberdaya.id/v1/trackDocumentOffice?filePath=${filePath}&documentId=${documentId}`,
                coEditing: { mode: "fast", change: true },
                lang: "en",
                mode: isViewerOnly ? "view" : "edit",
                user: {
                    id: user.id,
                    image: user.photoURL || "",
                    name: user.name,
                },
            },
        };
        // Generate JWT token
        const token = jsonwebtoken_1.default.sign(onlyOfficeConfig, process.env.JWT_SECRET_KEY_ONLYOFFICE || "", {});
        const signUrl = (url) => __awaiter(void 0, void 0, void 0, function* () {
            const storageFile = firebase_admin_1.bucket.file(url); // Assuming the 'url' is the file path in GCS
            const [signedUrl] = yield storageFile.getSignedUrl({
                action: "read",
                expires: Date.now() + 1000 * 60 * 60 * 24 * 7, // 7 days expiration
            });
            return signedUrl;
        });
        const combinedHistoryWithTokens = yield Promise.all(((documentData === null || documentData === void 0 ? void 0 : documentData.history) || []).map((historyEntry) => __awaiter(void 0, void 0, void 0, function* () {
            // Sign the URLs for bot/h 'url' and 'previous.url'
            var _a, _b;
            const signedUrl = yield signUrl(historyEntry.historyData.url);
            let signedPreviousUrl;
            if ((_b = (_a = historyEntry === null || historyEntry === void 0 ? void 0 : historyEntry.historyData) === null || _a === void 0 ? void 0 : _a.previous) === null || _b === void 0 ? void 0 : _b.url) {
                signedPreviousUrl = yield signUrl(historyEntry.historyData.previous.url);
            }
            let signedChangesUrl = null;
            if (historyEntry.historyData.changesUrl) {
                signedChangesUrl = yield signUrl(historyEntry.historyData.changesUrl);
            }
            let updatedHistoryData = Object.assign(Object.assign({}, historyEntry.historyData), { url: signedUrl, changesUrl: signedChangesUrl, previous: Object.assign(Object.assign({}, historyEntry.historyData.previous), { url: signedPreviousUrl }) });
            const token = jsonwebtoken_1.default.sign(updatedHistoryData, process.env.JWT_SECRET_KEY_ONLYOFFICE || "", {});
            // Return the combined object with history and token, and replace URLs with signed URLs
            return Object.assign(Object.assign({}, historyEntry), { historyData: Object.assign(Object.assign({}, updatedHistoryData), { token }) });
        })));
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document retrieved successfully.",
            data: Object.assign(Object.assign({}, onlyOfficeConfig), { token, currentVersion: documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion, documentHistory: combinedHistoryWithTokens }),
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.viewDocumentOffice = viewDocumentOffice;
const trackDocumentOffice = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { documentId } = req.query; // Extract documentId from query params
        const { status, url, history } = req.body; // Extract status and url from request body
        // Validate inputs
        if (typeof documentId !== "string" || !documentId) {
            return res.status(400).json({ error: "Invalid or missing documentId." });
        }
        // Fetch document details to retrieve the filePath
        const documentRef = firebase_admin_1.adminDb.collection("documents").doc(documentId);
        const documentSnapshot = yield documentRef.get();
        if (!documentSnapshot.exists) {
            return res.status(404).json({ error: "Document not found." });
        }
        const documentData = documentSnapshot.data();
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        if (!filePath) {
            return res
                .status(400)
                .json({ error: "Document storage path not found." });
        }
        switch (status) {
            case 2: // Document saved successfully
                if (!url) {
                    return res
                        .status(400)
                        .json({ error: "File URL is required for status 2." });
                }
                // Fetch the file from the provided URL
                const response = yield fetch(url);
                if (!response.ok) {
                    return res
                        .status(400)
                        .json({ error: "Failed to fetch the updated file from URL." });
                }
                const fileBuffer = yield response.arrayBuffer();
                const currentVersion = (documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion) || 1;
                const newVersion = currentVersion + 1;
                const baseFilePath = filePath.replace(/\/v\d+\/.*$/, ""); // Remove existing version from the path
                const newFilePath = `${baseFilePath}/v${newVersion}/${path_1.default.basename(filePath)}`;
                // Save the new file version in Firebase Storage
                const versionedStorageFile = firebase_admin_1.bucket.file(newFilePath);
                yield versionedStorageFile.save(Buffer.from(fileBuffer), {
                    resumable: false,
                    contentType: response.headers.get("content-type") || undefined, // Preserve content type
                });
                const changesUrl = req.body.changesurl;
                let changesFilePath = null;
                if (changesUrl) {
                    const changesResponse = yield fetch(changesUrl);
                    if (changesResponse.ok) {
                        const changesBuffer = yield changesResponse.arrayBuffer();
                        changesFilePath = `${baseFilePath}/v${newVersion}/changes.zip`;
                        const changesStorageFile = firebase_admin_1.bucket.file(changesFilePath);
                        yield changesStorageFile.save(Buffer.from(changesBuffer), {
                            resumable: false,
                            contentType: changesResponse.headers.get("content-type") || undefined,
                        });
                        // Don't make public
                    }
                    else {
                        console.warn("Failed to fetch changes file from changesUrl.");
                    }
                }
                const previousKey = documentData === null || documentData === void 0 ? void 0 : documentData.key;
                const newKey = generateDocumentKey();
                const previousHistory = (documentData === null || documentData === void 0 ? void 0 : documentData.history) || [];
                const newHistoryEntry = {
                    key: previousKey,
                    created: formatDateIndonesian(new Date()),
                    serverVersion: history === null || history === void 0 ? void 0 : history.serverVersion,
                    changes: (history === null || history === void 0 ? void 0 : history.changes) || [],
                    user: ((_b = (_a = history === null || history === void 0 ? void 0 : history.changes) === null || _a === void 0 ? void 0 : _a[0]) === null || _b === void 0 ? void 0 : _b.user) || {
                        id: "unknown",
                        name: "Unknown User",
                    },
                    historyData: {
                        changesUrl: changesFilePath,
                        fileType: path_1.default.extname(filePath).replace(".", ""),
                        key: newKey,
                        previous: {
                            fileType: path_1.default.extname(filePath).replace(".", ""),
                            key: previousKey,
                            url: `${filePath}`, // Previous file path
                        },
                        url: `${newFilePath}`, // New file path
                        version: newVersion,
                    },
                    version: newVersion, // Previous version
                };
                const updatedHistory = [...previousHistory, newHistoryEntry];
                yield documentRef.update({
                    key: newKey,
                    updatedAt: new Date().toISOString(),
                    currentVersion: newVersion,
                    storagePath: newFilePath,
                    history: updatedHistory,
                });
                console.log(`Document saved as a new version. New path: ${newFilePath}, Old path: ${filePath}`);
                break;
            case 1:
                console.log("Status 1 Document edited");
                break;
            case 3:
                console.log("Status 3 document saving error has occurred");
                break;
            case 4:
                console.log("Status 4 document is closed with no changes");
                break;
            case 6:
                console.log("Status 6 document is being edited, but the current document state is saved");
                break;
            case 7:
                console.log("Status 7 error has occurred while force saving the document.");
                break;
            default:
                console.log("Unhandled status:", status);
                break;
        }
        return res.json({
            error: 0,
            message: "TrackDocumentOffice event processed successfully.",
        });
    }
    catch (error) {
        console.error("Error in trackDocumentOffice:", error);
        return res.status(500).json({ error: "Internal Server Error" });
    }
});
exports.trackDocumentOffice = trackDocumentOffice;
const restoreDocumentOffice = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { documentId } = req.params;
        let { version } = req.body;
        version = Number(version);
        // Validate inputs
        if (!documentId || isNaN(version)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Invalid documentId or version.",
            });
        }
        // Fetch the document details
        const documentRef = firebase_admin_1.adminDb.collection("documents").doc(documentId);
        const documentSnapshot = yield documentRef.get();
        if (!documentSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document not found.",
            });
        }
        const documentData = documentSnapshot.data();
        const history = (documentData === null || documentData === void 0 ? void 0 : documentData.history) || [];
        // Find the desired version in the history
        const targetHistory = history.find((entry) => entry.version === version);
        if (!targetHistory) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Version not found in document history.",
            });
        }
        // Retrieve the target file from storage
        const targetFilePath = targetHistory.historyData.url;
        const storageFile = firebase_admin_1.bucket.file(targetFilePath);
        const [fileExists] = yield storageFile.exists();
        if (!fileExists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Target file for the specified version not found in storage.",
            });
        }
        // Read the target file contents
        const [fileBuffer] = yield storageFile.download();
        // Increment the version
        const currentVersion = (documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion) || 1;
        const newVersion = currentVersion + 1;
        // Generate a new file path
        const baseFilePath = targetFilePath.replace(/\/v\d+\/.*$/, ""); // Remove existing version details from the path
        const newFilePath = `${baseFilePath}/v${newVersion}/${path_1.default.basename(targetFilePath)}`;
        // Save the new file version to storage
        const newStorageFile = firebase_admin_1.bucket.file(newFilePath);
        yield newStorageFile.save(fileBuffer, {
            resumable: false,
            contentType: yield storageFile
                .getMetadata()
                .then((meta) => meta[0].contentType),
        });
        // Generate a new key for the new version
        const newKey = generateDocumentKey();
        // Add a new history entry
        const newHistoryEntry = {
            key: newKey,
            created: formatDateIndonesian(new Date()), // New creation date
            serverVersion: targetHistory.serverVersion || null, // Retain the original serverVersion
            changes: targetHistory.changes || [], // Retain changes from the target version
            user: targetHistory.user || { id: "unknown", name: "Unknown User" },
            historyData: {
                fileType: path_1.default.extname(targetFilePath).replace(".", ""),
                key: newKey,
                previous: {
                    fileType: path_1.default.extname(targetFilePath).replace(".", ""),
                    key: documentData === null || documentData === void 0 ? void 0 : documentData.key,
                    url: documentData === null || documentData === void 0 ? void 0 : documentData.storagePath,
                },
                url: newFilePath,
                version: newVersion,
            },
            version: newVersion,
        };
        const updatedHistory = [...history, newHistoryEntry];
        // Update the document data in Firestore
        yield documentRef.update({
            key: newKey,
            updatedAt: new Date().toISOString(),
            currentVersion: newVersion,
            storagePath: newFilePath,
            history: updatedHistory,
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document copied to a new version successfully.",
            data: {
                currentVersion: newVersion,
                newFilePath,
            },
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.restoreDocumentOffice = restoreDocumentOffice;
const convertDocumentOfficePDF = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { documentId } = req.params;
        const user = req.user;
        if (!documentId) {
            return res.status(400).json({
                status: "error",
                message: "Missing required parameter: documentId.",
            });
        }
        // Fetch original document data
        const documentSnapshot = yield firebase_admin_1.adminDb
            .collection("documents")
            .doc(documentId)
            .get();
        if (!documentSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                message: "Document not found.",
            });
        }
        const documentData = documentSnapshot.data();
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        const fileType = (documentData === null || documentData === void 0 ? void 0 : documentData.fileType) || "docx";
        const documentTitle = (documentData === null || documentData === void 0 ? void 0 : documentData.title) || "Untitled Document";
        if (!filePath) {
            return res.status(500).json({
                status: "error",
                message: "Document storage path is missing in the database.",
            });
        }
        // Generate a signed URL for the document
        const storageFile = firebase_admin_1.bucket.file(filePath);
        const [signedUrl] = yield storageFile.getSignedUrl({
            action: "read",
            expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
        });
        // Generate unique key for conversion
        const key = generateDocumentKey();
        // Prepare payload for ONLYOFFICE conversion
        const payload = {
            async: false,
            filetype: fileType,
            key,
            outputtype: "pdf",
            title: documentTitle,
            url: signedUrl,
        };
        const token = jsonwebtoken_1.default.sign(payload, process.env.JWT_SECRET_KEY_ONLYOFFICE || "", {});
        const requestPayload = Object.assign(Object.assign({}, payload), { token });
        const documentServerUrl = process.env.ONLYOFFICE_DOCUMENT_SERVER_URL || "";
        const conversionUrl = `${documentServerUrl}?shardkey=${key}`;
        // Make conversion request
        const response = yield axios_1.default.post(conversionUrl, requestPayload, {
            headers: { "Content-Type": "application/json" },
        });
        const conversionResult = response.data;
        if (conversionResult.error || !conversionResult.endConvert) {
            return res.status(500).json({
                status: "error",
                message: `Conversion failed: ${conversionResult.error}`,
            });
        }
        const pdfUrl = conversionResult.fileUrl;
        // Fetch the PDF
        const pdfResponse = yield fetch(pdfUrl);
        if (!pdfResponse.ok) {
            return res.status(500).json({
                status: "error",
                message: "Failed to download converted PDF from ONLYOFFICE.",
            });
        }
        const pdfBuffer = yield pdfResponse.arrayBuffer();
        const fileSizeMB = (pdfBuffer.byteLength / (1024 * 1024)).toFixed(2); // Calculate file size in MB
        // Generate the new file path
        const baseFilePath = filePath.replace(/\/[^/]+$/, "");
        const pdfFilePath = `${baseFilePath}/pdf/${documentTitle}.pdf`;
        // Save the PDF to Firebase Storage
        const pdfStorageFile = firebase_admin_1.bucket.file(pdfFilePath);
        yield pdfStorageFile.save(Buffer.from(pdfBuffer), {
            resumable: false,
            contentType: "application/pdf",
        });
        const newDocumentRef = firebase_admin_1.adminDb.collection("documents").doc();
        const newDocumentData = {
            title: `${documentTitle} (Converted PDF)`,
            id: newDocumentRef.id,
            access: documentData.access,
            categoryId: (documentData === null || documentData === void 0 ? void 0 : documentData.categoryId) || null,
            categoryName: (documentData === null || documentData === void 0 ? void 0 : documentData.categoryName) || null,
            currentVersion: 1,
            createdBy: user === null || user === void 0 ? void 0 : user.id,
            createdByEmail: user === null || user === void 0 ? void 0 : user.email,
            createdByName: user === null || user === void 0 ? void 0 : user.name,
            createdByPhotoURL: user === null || user === void 0 ? void 0 : user.photoURL,
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
            updatedBy: user === null || user === void 0 ? void 0 : user.id,
            storagePath: pdfFilePath,
            fileExtension: "pdf",
            status: (documentData === null || documentData === void 0 ? void 0 : documentData.status) || null,
            tags: (documentData === null || documentData === void 0 ? void 0 : documentData.tags) || null,
            workspaceId: documentData === null || documentData === void 0 ? void 0 : documentData.workspaceId,
            divisiFolderId: documentData === null || documentData === void 0 ? void 0 : documentData.divisiFolderId,
            folderName: (documentData === null || documentData === void 0 ? void 0 : documentData.folderName) || null,
            key,
            fileSizeMB: parseFloat(fileSizeMB), // Store file size in MB
            history: [
                {
                    version: 1,
                    created: formatDateIndonesian(new Date()),
                    key,
                    user: {
                        id: user === null || user === void 0 ? void 0 : user.id,
                        name: user === null || user === void 0 ? void 0 : user.name,
                    },
                    historyData: {
                        fileType: "pdf",
                        key,
                        url: pdfFilePath,
                    },
                },
            ],
        };
        // Save to a new collection
        yield newDocumentRef.set(newDocumentData);
        return res.status(200).json({
            status: "success",
            message: "Document converted to PDF and stored successfully.",
            data: { newDocumentId: newDocumentRef.id, pdfFilePath, fileSizeMB },
        });
    }
    catch (error) {
        console.error("Error in convertDocumentOfficePDF:", error);
        // Handle axios-specific error responses
        if (axios_1.default.isAxiosError(error) && error.response) {
            return res.status(error.response.status).json({
                status: "error",
                message: ((_a = error.response.data) === null || _a === void 0 ? void 0 : _a.error) || "Error from ONLYOFFICE service",
            });
        }
        return res.status(500).json({
            status: "error",
            message: "Internal Server Error during document conversion.",
        });
    }
});
exports.convertDocumentOfficePDF = convertDocumentOfficePDF;
const viewRekamanOffice = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { rekamanId, workspaceId, rekamanFolderId } = req.params;
        const user = req.user;
        // Fetch workspace, folder, and document data in parallel
        const [workspaceSnapshot, folderSnapshot, rekamanSnapshot] = yield Promise.all([
            firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId).get(),
            firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("rekamanFolders")
                .doc(rekamanFolderId)
                .get(),
            firebase_admin_1.adminDb.collection("rekaman").doc(rekamanId).get(),
        ]);
        // Validate workspace
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found or access denied.",
            });
        }
        // Validate folder access
        if (!folderSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Rekaman Folder not found.",
            });
        }
        const folderData = folderSnapshot.data();
        const folderAllowedRoles = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles) || [];
        const folderAssignedUsers = (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers) || [];
        const folderRequiresApproval = folderData === null || folderData === void 0 ? void 0 : folderData.access.requiresApproval;
        const folderOwner = (folderData === null || folderData === void 0 ? void 0 : folderData.access.owenrId) == user.id;
        const hasFolderAccess = folderAllowedRoles.includes(user.role) ||
            folderAssignedUsers.some((assignedUser) => assignedUser.id === user.id);
        const isAdminOrSuperadmin = user.role === "superadmin" || user.role === "admin";
        if (!(hasFolderAccess ||
            folderRequiresApproval ||
            isAdminOrSuperadmin ||
            folderOwner)) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access to this rekaman folder is denied.",
            });
        }
        // Validate document access
        if (!rekamanSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Rekaman not found." });
        }
        const documentData = rekamanSnapshot.data();
        const documentAllowedRoles = (documentData === null || documentData === void 0 ? void 0 : documentData.access.allowedRoles) || [];
        const documentAssignedUsers = (documentData === null || documentData === void 0 ? void 0 : documentData.access.assignedUsers) || [];
        const documentOwner = (documentData === null || documentData === void 0 ? void 0 : documentData.access.ownerId) == user.id;
        const invitedUser = documentAssignedUsers.find((invited) => invited.id === user.id);
        const isViewerOnly = (invitedUser === null || invitedUser === void 0 ? void 0 : invitedUser.role) === "viewer";
        const hasDocumentAccess = documentAllowedRoles.includes(user.role) || invitedUser || documentOwner;
        if (!hasDocumentAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to view this rekaman.",
            });
        }
        // Ensure the document has a unique key
        let documentKey = documentData === null || documentData === void 0 ? void 0 : documentData.key;
        if (!documentKey) {
            documentKey = generateDocumentKey();
            yield firebase_admin_1.adminDb
                .collection("rekaman")
                .doc(rekamanId)
                .update({ key: documentKey });
        }
        // Generate a signed URL for the document
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        const storageFile = firebase_admin_1.bucket.file(filePath);
        const [url] = yield storageFile.getSignedUrl({
            action: "read",
            expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
        });
        // Determine file type
        const extension = path_1.default.extname(filePath).toLowerCase().replace(".", "");
        const fileType = extension || "unknown";
        const documentType = determineDocumentType(fileType);
        // Prepare OnlyOffice configuration
        const onlyOfficeConfig = {
            document: {
                fileType,
                info: {
                    favorite: false,
                    folder: (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || "Unknown Folder",
                    owner: ((_b = documentData === null || documentData === void 0 ? void 0 : documentData.access) === null || _b === void 0 ? void 0 : _b.ownerName) || "Unknown",
                    uploaded: (documentData === null || documentData === void 0 ? void 0 : documentData.createdAt) || "Unknown",
                },
                key: documentKey,
                title: (documentData === null || documentData === void 0 ? void 0 : documentData.title) || "Untitled Document",
                url,
                permissions: {
                    edit: !isViewerOnly,
                    download: !isViewerOnly,
                    print: !isViewerOnly,
                    chat: true,
                    comment: true,
                    copy: !isViewerOnly,
                    fillForms: true,
                    review: true,
                    modifyFilter: false,
                    modifyContentControl: false,
                    protect: false,
                },
            },
            documentType,
            editorConfig: {
                callbackUrl: `https://dmstestapi.mitraberdaya.id/v1/trackRekamanOffice?filePath=${filePath}&rekamanId=${rekamanId}`,
                coEditing: { mode: "fast", change: true },
                lang: "en",
                mode: isViewerOnly ? "view" : "edit",
                user: {
                    id: user.id,
                    image: user.photoURL || "",
                    name: user.name,
                },
            },
        };
        // Generate JWT token
        const token = jsonwebtoken_1.default.sign(onlyOfficeConfig, process.env.JWT_SECRET_KEY_ONLYOFFICE || "", {});
        const signUrl = (url) => __awaiter(void 0, void 0, void 0, function* () {
            const storageFile = firebase_admin_1.bucket.file(url); // Assuming the 'url' is the file path in GCS
            const [signedUrl] = yield storageFile.getSignedUrl({
                action: "read",
                expires: Date.now() + 1000 * 60 * 60 * 24 * 7, // 7 days expiration
            });
            return signedUrl;
        });
        const combinedHistoryWithTokens = yield Promise.all(((documentData === null || documentData === void 0 ? void 0 : documentData.history) || []).map((historyEntry) => __awaiter(void 0, void 0, void 0, function* () {
            // Sign the URLs for bot/h 'url' and 'previous.url'
            var _a, _b;
            const signedUrl = yield signUrl(historyEntry.historyData.url);
            let signedPreviousUrl;
            if ((_b = (_a = historyEntry === null || historyEntry === void 0 ? void 0 : historyEntry.historyData) === null || _a === void 0 ? void 0 : _a.previous) === null || _b === void 0 ? void 0 : _b.url) {
                signedPreviousUrl = yield signUrl(historyEntry.historyData.previous.url);
            }
            let signedChangesUrl = null;
            if (historyEntry.historyData.changesUrl) {
                signedChangesUrl = yield signUrl(historyEntry.historyData.changesUrl);
            }
            let updatedHistoryData = Object.assign(Object.assign({}, historyEntry.historyData), { url: signedUrl, changesUrl: signedChangesUrl, previous: Object.assign(Object.assign({}, historyEntry.historyData.previous), { url: signedPreviousUrl }) });
            const token = jsonwebtoken_1.default.sign(updatedHistoryData, process.env.JWT_SECRET_KEY_ONLYOFFICE || "", {});
            // Return the combined object with history and token, and replace URLs with signed URLs
            return Object.assign(Object.assign({}, historyEntry), { historyData: Object.assign(Object.assign({}, updatedHistoryData), { token }) });
        })));
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Rekaman retrieved successfully.",
            data: Object.assign(Object.assign({}, onlyOfficeConfig), { token, currentVersion: documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion, documentHistory: combinedHistoryWithTokens }),
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.viewRekamanOffice = viewRekamanOffice;
const trackRekamanOffice = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { rekamanId } = req.query; // Extract rekamanId from query params
        const { status, url, history } = req.body; // Extract status and url from request body
        // Validate inputs
        if (typeof rekamanId !== "string" || !rekamanId) {
            return res.status(400).json({ error: "Invalid or missing rekamanId." });
        }
        // Fetch document details to retrieve the filePath
        const rekamanRef = firebase_admin_1.adminDb.collection("rekaman").doc(rekamanId);
        const rekamanSnapshot = yield rekamanRef.get();
        if (!rekamanSnapshot.exists) {
            return res.status(404).json({ error: "Rekaman not found." });
        }
        const documentData = rekamanSnapshot.data();
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        if (!filePath) {
            return res.status(400).json({ error: "Rekaman storage path not found." });
        }
        switch (status) {
            case 2: // Document saved successfully
                if (!url) {
                    return res
                        .status(400)
                        .json({ error: "File URL is required for status 2." });
                }
                // Fetch the file from the provided URL
                const response = yield fetch(url);
                if (!response.ok) {
                    return res
                        .status(400)
                        .json({ error: "Failed to fetch the updated file from URL." });
                }
                const fileBuffer = yield response.arrayBuffer();
                const currentVersion = (documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion) || 1;
                const newVersion = currentVersion + 1;
                const baseFilePath = filePath.replace(/\/v\d+\/.*$/, ""); // Remove existing version from the path
                const newFilePath = `${baseFilePath}/v${newVersion}/${path_1.default.basename(filePath)}`;
                // Save the new file version in Firebase Storage
                const versionedStorageFile = firebase_admin_1.bucket.file(newFilePath);
                yield versionedStorageFile.save(Buffer.from(fileBuffer), {
                    resumable: false,
                    contentType: response.headers.get("content-type") || undefined, // Preserve content type
                });
                const changesUrl = req.body.changesurl;
                let changesFilePath = null;
                if (changesUrl) {
                    const changesResponse = yield fetch(changesUrl);
                    if (changesResponse.ok) {
                        const changesBuffer = yield changesResponse.arrayBuffer();
                        changesFilePath = `${baseFilePath}/v${newVersion}/changes.zip`;
                        const changesStorageFile = firebase_admin_1.bucket.file(changesFilePath);
                        yield changesStorageFile.save(Buffer.from(changesBuffer), {
                            resumable: false,
                            contentType: changesResponse.headers.get("content-type") || undefined,
                        });
                        // Don't make public
                    }
                    else {
                        console.warn("Failed to fetch changes file from changesUrl.");
                    }
                }
                const previousKey = documentData === null || documentData === void 0 ? void 0 : documentData.key;
                const newKey = generateDocumentKey();
                const previousHistory = (documentData === null || documentData === void 0 ? void 0 : documentData.history) || [];
                const newHistoryEntry = {
                    key: previousKey,
                    created: formatDateIndonesian(new Date()),
                    serverVersion: history === null || history === void 0 ? void 0 : history.serverVersion,
                    changes: (history === null || history === void 0 ? void 0 : history.changes) || [],
                    user: ((_b = (_a = history === null || history === void 0 ? void 0 : history.changes) === null || _a === void 0 ? void 0 : _a[0]) === null || _b === void 0 ? void 0 : _b.user) || {
                        id: "unknown",
                        name: "Unknown User",
                    },
                    historyData: {
                        changesUrl: changesFilePath,
                        fileType: path_1.default.extname(filePath).replace(".", ""),
                        key: newKey,
                        previous: {
                            fileType: path_1.default.extname(filePath).replace(".", ""),
                            key: previousKey,
                            url: `${filePath}`, // Previous file path
                        },
                        url: `${newFilePath}`, // New file path
                        version: newVersion,
                    },
                    version: newVersion, // Previous version
                };
                const updatedHistory = [...previousHistory, newHistoryEntry];
                yield rekamanRef.update({
                    key: newKey,
                    updatedAt: new Date().toISOString(),
                    currentVersion: newVersion,
                    storagePath: newFilePath,
                    history: updatedHistory,
                });
                console.log(`Document saved as a new version. New path: ${newFilePath}, Old path: ${filePath}`);
                break;
            case 1:
                console.log("Status 1 Document edited");
                break;
            case 3:
                console.log("Status 3 document saving error has occurred");
                break;
            case 4:
                console.log("Status 4 document is closed with no changes");
                break;
            case 6:
                console.log("Status 6 document is being edited, but the current document state is saved");
                break;
            case 7:
                console.log("Status 7 error has occurred while force saving the document.");
                break;
            default:
                console.log("Unhandled status:", status);
                break;
        }
        return res.json({
            error: 0,
            message: "TrackDocumentOffice event processed successfully.",
        });
    }
    catch (error) {
        console.error("Error in trackDocumentOffice:", error);
        return res.status(500).json({ error: "Internal Server Error" });
    }
});
exports.trackRekamanOffice = trackRekamanOffice;
const restoreRekamanOffice = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { rekamanId } = req.params;
        let { version } = req.body;
        version = Number(version);
        // Validate inputs
        if (!rekamanId || isNaN(version)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Invalid rekamanId or version.",
            });
        }
        // Fetch the document details
        const rekamanRef = firebase_admin_1.adminDb.collection("rekaman").doc(rekamanId);
        const rekamanSnapshot = yield rekamanRef.get();
        if (!rekamanSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document not found.",
            });
        }
        const documentData = rekamanSnapshot.data();
        const history = (documentData === null || documentData === void 0 ? void 0 : documentData.history) || [];
        // Find the desired version in the history
        const targetHistory = history.find((entry) => entry.version === version);
        if (!targetHistory) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Version not found in document history.",
            });
        }
        // Retrieve the target file from storage
        const targetFilePath = targetHistory.historyData.url;
        const storageFile = firebase_admin_1.bucket.file(targetFilePath);
        const [fileExists] = yield storageFile.exists();
        if (!fileExists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Target file for the specified version not found in storage.",
            });
        }
        // Read the target file contents
        const [fileBuffer] = yield storageFile.download();
        // Increment the version
        const currentVersion = (documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion) || 1;
        const newVersion = currentVersion + 1;
        // Generate a new file path
        const baseFilePath = targetFilePath.replace(/\/v\d+\/.*$/, ""); // Remove existing version details from the path
        const newFilePath = `${baseFilePath}/v${newVersion}/${path_1.default.basename(targetFilePath)}`;
        // Save the new file version to storage
        const newStorageFile = firebase_admin_1.bucket.file(newFilePath);
        yield newStorageFile.save(fileBuffer, {
            resumable: false,
            contentType: yield storageFile
                .getMetadata()
                .then((meta) => meta[0].contentType),
        });
        // Generate a new key for the new version
        const newKey = generateDocumentKey();
        // Add a new history entry
        const newHistoryEntry = {
            key: newKey,
            created: formatDateIndonesian(new Date()), // New creation date
            serverVersion: targetHistory.serverVersion || null, // Retain the original serverVersion
            changes: targetHistory.changes || [], // Retain changes from the target version
            user: targetHistory.user || { id: "unknown", name: "Unknown User" },
            historyData: {
                fileType: path_1.default.extname(targetFilePath).replace(".", ""),
                key: newKey,
                previous: {
                    fileType: path_1.default.extname(targetFilePath).replace(".", ""),
                    key: documentData === null || documentData === void 0 ? void 0 : documentData.key,
                    url: documentData === null || documentData === void 0 ? void 0 : documentData.storagePath,
                },
                url: newFilePath,
                version: newVersion,
            },
            version: newVersion,
        };
        const updatedHistory = [...history, newHistoryEntry];
        // Update the document data in Firestore
        yield rekamanRef.update({
            key: newKey,
            updatedAt: new Date().toISOString(),
            currentVersion: newVersion,
            storagePath: newFilePath,
            history: updatedHistory,
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Rekaman copied to a new version successfully.",
            data: {
                currentVersion: newVersion,
                newFilePath,
            },
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.restoreRekamanOffice = restoreRekamanOffice;
const createNewFileFromTemplate = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { workspaceId } = req.params;
        const parsedBody = onlyoffice_1.createDocumentSchema.parse(req.body);
        const { title, categoryId, status, tags, divisiFolderId, extension } = parsedBody;
        const user = req.user;
        // Validate required fields
        if (!title || !divisiFolderId || !extension) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Missing required fields (title, divisiFolderId, extension).",
            });
        }
        // Validate workspace
        const workspaceRef = firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId);
        const workspaceSnapshot = yield workspaceRef.get();
        if (!workspaceSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceSnapshot.data();
        if (workspaceData.companyId !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied.",
            });
        }
        const categoryRef = firebase_admin_1.adminDb.collection("categories").doc(categoryId);
        const categorySnapshot = yield categoryRef.get();
        if (!categorySnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Category not found.",
            });
        }
        const categoryData = categorySnapshot.data();
        // Validate folder
        const folderRef = workspaceRef.collection("folders").doc(divisiFolderId);
        const folderSnapshot = yield folderRef.get();
        if (!folderSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Folder not found.",
            });
        }
        const folderData = folderSnapshot.data();
        const hasAccess = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles.includes(user.role)) ||
            (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers.some((assignedUser) => assignedUser.id === user.id)) ||
            ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.ownerId) === user.id;
        if (!hasAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied to this folder.",
            });
        }
        // Determine the template file to duplicate
        const templateFileName = `master ${extension}.${extension}`;
        const templatePath = `Master/Template New File/${templateFileName}`;
        // Fetch template from Firebase Storage
        const [templateBuffer] = yield firebase_admin_1.bucket.file(templatePath).download();
        if (!templateBuffer) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: `Template file ${templateFileName} not found.`,
            });
        }
        // Calculate file size and validate against storage limits
        const fileSizeMB = templateBuffer.length / (1024 * 1024);
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user.companyId);
        const companySnapshot = yield companyRef.get();
        const companyData = companySnapshot.data();
        const { storageLimit = 0 } = (companyData === null || companyData === void 0 ? void 0 : companyData.plan) || {};
        const { storageUsage = 0 } = (companyData === null || companyData === void 0 ? void 0 : companyData.usage) || {};
        if (storageUsage + fileSizeMB > storageLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Current usage is ${storageUsage} MB, plan allows ${storageLimit} MB.`,
            });
        }
        // Define the new file path in Firebase Storage
        const newFilePath = `${user.companyId}/workspaces/${workspaceId}/folders/${divisiFolderId}/${title}.${extension}`;
        // Save the file to the new location
        const newFile = firebase_admin_1.bucket.file(newFilePath);
        yield newFile.save(templateBuffer);
        // Update company storage usage
        yield companyRef.update({
            "usage.storageUsage": firestore_1.FieldValue.increment(fileSizeMB),
        });
        // Save metadata in Firestore
        const docRef = firebase_admin_1.adminDb.collection("documents").doc();
        let newKey = generateDocumentKey();
        const historyEntry = {
            created: formatDateIndonesian(new Date()),
            key: newKey,
            user: {
                id: user.id,
                name: user.name || "Unknown User",
            },
            historyData: {
                fileType: extension,
                key: newKey,
                url: newFilePath,
            },
            version: 1,
        };
        const documentData = {
            id: docRef.id,
            workspaceId,
            divisiFolderId,
            folderName: folderData.folderName,
            title,
            storagePath: newFilePath,
            fileExtension: extension,
            fileSizeMB,
            createdBy: user.id,
            createdByEmail: user === null || user === void 0 ? void 0 : user.email,
            createdByName: user === null || user === void 0 ? void 0 : user.name,
            createdByPhotoURL: user === null || user === void 0 ? void 0 : user.photoURL,
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
            status,
            categoryId,
            categoryName: categoryData.categoryName,
            history: [historyEntry],
            access: {
                allowedRoles: ["admin", "superadmin"],
                ownerId: user.id,
                ownerName: user.name,
            },
            currentVersion: 1,
            tags,
            key: newKey,
        };
        yield docRef.set(documentData);
        return res.status(201).json({
            status: "success",
            message: "New file created successfully.",
            data: documentData,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
            error: error.message,
        });
    }
});
exports.createNewFileFromTemplate = createNewFileFromTemplate;
const createNewRekamanFileFromTemplate = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { workspaceId } = req.params;
        const parsedBody = onlyoffice_1.createRekamanSchema.parse(req.body);
        const { title, rekamanFolderId, extension } = parsedBody;
        const user = req.user;
        // Validate required fields
        if (!title || !rekamanFolderId || !extension) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Missing required fields (title, divisiFolderId, extension).",
            });
        }
        // Validate workspace
        const workspaceRef = firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId);
        const workspaceSnapshot = yield workspaceRef.get();
        if (!workspaceSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceSnapshot.data();
        if (workspaceData.companyId !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied.",
            });
        }
        // Validate folder
        const folderRef = workspaceRef
            .collection("rekamanFolders")
            .doc(rekamanFolderId);
        const folderSnapshot = yield folderRef.get();
        if (!folderSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Rekaman Folder not found.",
            });
        }
        const folderData = folderSnapshot.data();
        const hasAccess = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles.includes(user.role)) ||
            (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers.some((assignedUser) => assignedUser.id === user.id)) ||
            ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.ownerId) === user.id;
        if (!hasAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied to this folder.",
            });
        }
        // Determine the template file to duplicate
        const templateFileName = `master ${extension}.${extension}`;
        const templatePath = `Master/Template New File/${templateFileName}`;
        // Fetch template from Firebase Storage
        const [templateBuffer] = yield firebase_admin_1.bucket.file(templatePath).download();
        if (!templateBuffer) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: `Template file ${templateFileName} not found.`,
            });
        }
        // Calculate file size and validate against storage limits
        const fileSizeMB = templateBuffer.length / (1024 * 1024);
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user.companyId);
        const companySnapshot = yield companyRef.get();
        const companyData = companySnapshot.data();
        const { storageLimit = 0 } = (companyData === null || companyData === void 0 ? void 0 : companyData.plan) || {};
        const { storageUsage = 0 } = (companyData === null || companyData === void 0 ? void 0 : companyData.usage) || {};
        if (storageUsage + fileSizeMB > storageLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Current usage is ${storageUsage} MB, plan allows ${storageLimit} MB.`,
            });
        }
        const newFilePath = `${user.companyId}/workspaces/${workspaceId}/rekamanFolders/${rekamanFolderId}/${title}.${extension}`;
        // Save the file to the new location
        const newFile = firebase_admin_1.bucket.file(newFilePath);
        yield newFile.save(templateBuffer);
        // Update company storage usage
        yield companyRef.update({
            "usage.storageUsage": firestore_1.FieldValue.increment(fileSizeMB),
        });
        // Save metadata in Firestore
        const docRef = firebase_admin_1.adminDb.collection("rekaman").doc();
        let newKey = generateDocumentKey();
        const historyEntry = {
            created: formatDateIndonesian(new Date()),
            key: newKey,
            user: {
                id: user.id,
                name: user.name || "Unknown User",
            },
            historyData: {
                fileType: extension,
                key: newKey,
                url: newFilePath,
            },
            version: 1,
        };
        const documentData = {
            id: docRef.id,
            workspaceId,
            rekamanFolderId,
            folderName: folderData.folderName,
            title,
            storagePath: newFilePath,
            fileExtension: extension,
            fileSizeMB,
            createdBy: user.id,
            createdByEmail: user === null || user === void 0 ? void 0 : user.email,
            createdByName: user === null || user === void 0 ? void 0 : user.name,
            createdByPhotoURL: user === null || user === void 0 ? void 0 : user.photoURL,
            createdAt: new Date().toISOString(),
            history: [historyEntry],
            access: {
                allowedRoles: ["admin", "superadmin"],
                ownerId: user.id,
                ownerName: user.name,
            },
            currentVersion: 1,
            key: newKey,
        };
        yield docRef.set(documentData);
        return res.status(201).json({
            status: "success",
            message: "New file created successfully.",
            data: documentData,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
            error: error.message,
        });
    }
});
exports.createNewRekamanFileFromTemplate = createNewRekamanFileFromTemplate;
const viewDocumentApprovalOffice = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { documentApprovalId, workspaceId } = req.params;
        const user = req.user;
        // Fetch workspace, folder, and document data in parallel
        const [workspaceSnapshot, documentSnapshot] = yield Promise.all([
            firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId).get(),
            firebase_admin_1.adminDb.collection("documentApprovals").doc(documentApprovalId).get(),
        ]);
        // Validate workspace
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found or access denied.",
            });
        }
        // Validate document access
        if (!documentSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Document not found." });
        }
        const documentData = documentSnapshot.data();
        const documentAllowedRoles = (documentData === null || documentData === void 0 ? void 0 : documentData.access.allowedRoles) || [];
        const documentInvitedUsers = (documentData === null || documentData === void 0 ? void 0 : documentData.access.invitedUsers) || [];
        const documentOwner = (documentData === null || documentData === void 0 ? void 0 : documentData.access.owenrId) == user.id;
        const invitedUser = documentInvitedUsers.find((invited) => invited.id === user.id);
        const isViewerOnly = (invitedUser === null || invitedUser === void 0 ? void 0 : invitedUser.role) === "viewer";
        const hasDocumentAccess = documentAllowedRoles.includes(user.role) || invitedUser || documentOwner;
        if (!hasDocumentAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to view this document.",
            });
        }
        // Ensure the document has a unique key
        let documentKey = documentData === null || documentData === void 0 ? void 0 : documentData.key;
        if (!documentKey) {
            documentKey = generateDocumentKey();
            yield firebase_admin_1.adminDb
                .collection("documentApprovals")
                .doc(documentApprovalId)
                .update({ key: documentKey });
        }
        // Generate a signed URL for the document
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        const storageFile = firebase_admin_1.bucket.file(filePath);
        const [url] = yield storageFile.getSignedUrl({
            action: "read",
            expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
        });
        const onlyOfficeConfig = {
            document: {
                fileType: documentData === null || documentData === void 0 ? void 0 : documentData.fileExtension,
                info: {
                    favorite: false,
                    owner: ((_b = documentData === null || documentData === void 0 ? void 0 : documentData.access) === null || _b === void 0 ? void 0 : _b.ownerName) || "Unknown",
                    uploaded: (documentData === null || documentData === void 0 ? void 0 : documentData.createdAt) || "Unknown",
                },
                key: documentKey,
                title: (documentData === null || documentData === void 0 ? void 0 : documentData.title) || "Untitled Document",
                url,
                permissions: {
                    edit: !isViewerOnly,
                    download: !isViewerOnly,
                    print: !isViewerOnly,
                    chat: true,
                    comment: true,
                    copy: !isViewerOnly,
                    fillForms: true,
                    review: false,
                    modifyFilter: false,
                    modifyContentControl: false,
                    protect: false,
                },
            },
            documentType: documentData === null || documentData === void 0 ? void 0 : documentData.fileExtension,
            editorConfig: {
                callbackUrl: `https://dmstestapi.mitraberdaya.id/v1/trackDocumentApprovalOffice?filePath=${filePath}&documentApprovalId=${documentApprovalId}`,
                coEditing: { mode: "fast", change: true },
                lang: "en",
                mode: isViewerOnly ? "view" : "edit",
                user: {
                    id: user.id,
                    image: user.photoURL || "",
                    name: user.name,
                },
            },
        };
        // Generate JWT token
        const token = jsonwebtoken_1.default.sign(onlyOfficeConfig, process.env.JWT_SECRET_KEY_ONLYOFFICE || "", {});
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document Approval retrieved successfully.",
            data: Object.assign(Object.assign({}, onlyOfficeConfig), { token, currentVersion: documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion }),
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.viewDocumentApprovalOffice = viewDocumentApprovalOffice;
const trackDocumentApprovalOffice = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { documentApprovalId } = req.query;
        const { status, url, history } = req.body;
        // Validate inputs
        if (typeof documentApprovalId !== "string" || !documentApprovalId) {
            return res.status(400).json({ error: "Invalid or missing documentId." });
        }
        // Fetch document details to retrieve the filePath
        const documentRef = firebase_admin_1.adminDb
            .collection("documentApprovals")
            .doc(documentApprovalId);
        const documentSnapshot = yield documentRef.get();
        if (!documentSnapshot.exists) {
            return res.status(404).json({ error: "Document Approval not found." });
        }
        const documentData = documentSnapshot.data();
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        if (!filePath) {
            return res
                .status(400)
                .json({ error: "Document storage path not found." });
        }
        switch (status) {
            case 2: // Document saved successfully
                if (!url) {
                    return res
                        .status(400)
                        .json({ error: "File URL is required for status 2." });
                }
                // Fetch the file from the provided URL
                const response = yield fetch(url);
                if (!response.ok) {
                    return res
                        .status(400)
                        .json({ error: "Failed to fetch the updated file from URL." });
                }
                const fileBuffer = yield response.arrayBuffer();
                const currentVersion = (documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion) || 1;
                const newVersion = currentVersion + 1;
                const baseFilePath = filePath.replace(/\/v\d+\/.*$/, ""); // Remove existing version from the path
                const newFilePath = `${baseFilePath}/v${newVersion}/${path_1.default.basename(filePath)}`;
                // Save the new file version in Firebase Storage
                const versionedStorageFile = firebase_admin_1.bucket.file(newFilePath);
                yield versionedStorageFile.save(Buffer.from(fileBuffer), {
                    resumable: false,
                    contentType: response.headers.get("content-type") || undefined, // Preserve content type
                });
                const changesUrl = req.body.changesurl;
                let changesFilePath = null;
                if (changesUrl) {
                    const changesResponse = yield fetch(changesUrl);
                    if (changesResponse.ok) {
                        const changesBuffer = yield changesResponse.arrayBuffer();
                        changesFilePath = `${baseFilePath}/v${newVersion}/changes.zip`;
                        const changesStorageFile = firebase_admin_1.bucket.file(changesFilePath);
                        yield changesStorageFile.save(Buffer.from(changesBuffer), {
                            resumable: false,
                            contentType: changesResponse.headers.get("content-type") || undefined,
                        });
                        // Don't make public
                    }
                    else {
                        console.warn("Failed to fetch changes file from changesUrl.");
                    }
                }
                const previousKey = documentData === null || documentData === void 0 ? void 0 : documentData.key;
                const newKey = generateDocumentKey();
                const previousHistory = (documentData === null || documentData === void 0 ? void 0 : documentData.history) || [];
                const newHistoryEntry = {
                    key: previousKey,
                    created: formatDateIndonesian(new Date()),
                    serverVersion: history === null || history === void 0 ? void 0 : history.serverVersion,
                    changes: (history === null || history === void 0 ? void 0 : history.changes) || [],
                    user: ((_b = (_a = history === null || history === void 0 ? void 0 : history.changes) === null || _a === void 0 ? void 0 : _a[0]) === null || _b === void 0 ? void 0 : _b.user) || {
                        id: "unknown",
                        name: "Unknown User",
                    },
                    historyData: {
                        changesUrl: changesFilePath,
                        fileType: path_1.default.extname(filePath).replace(".", ""),
                        key: newKey,
                        previous: {
                            fileType: path_1.default.extname(filePath).replace(".", ""),
                            key: previousKey,
                            url: `${filePath}`, // Previous file path
                        },
                        url: `${newFilePath}`, // New file path
                        version: newVersion,
                    },
                    version: newVersion, // Previous version
                };
                const updatedHistory = [...previousHistory, newHistoryEntry];
                yield documentRef.update({
                    key: newKey,
                    updatedAt: new Date().toISOString(),
                    currentVersion: newVersion,
                    storagePath: newFilePath,
                    history: updatedHistory,
                });
                console.log(`Document saved as a new version. New path: ${newFilePath}, Old path: ${filePath}`);
                break;
            case 1:
                console.log("Status 1 Document edited");
                break;
            case 3:
                console.log("Status 3 document saving error has occurred");
                break;
            case 4:
                console.log("Status 4 document is closed with no changes");
                break;
            case 6:
                console.log("Status 6 document is being edited, but the current document state is saved");
                break;
            case 7:
                console.log("Status 7 error has occurred while force saving the document.");
                break;
            default:
                console.log("Unhandled status:", status);
                break;
        }
        return res.json({
            error: 0,
            message: "TrackDocumentOffice event processed successfully.",
        });
    }
    catch (error) {
        console.error("Error in trackDocumentOffice:", error);
        return res.status(500).json({ error: "Internal Server Error" });
    }
});
exports.trackDocumentApprovalOffice = trackDocumentApprovalOffice;
