"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendNotificationRekamanAccess = exports.viewNotifications = exports.markAllNotificationsAsRead = exports.updateRecipientIsRead = exports.sendNotificationSingle = exports.sendNotificationAccess = exports.sendNotificationDocumentApproval = exports.sendNotificationFolderAccess = exports.sendNotification = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const sendNotification = (documentId_1, documentTitle_1, companyId_1, createdBy_1, createdByName_1, createdByEmail_1, createdByPhotoURL_1, workspaceId_1, divisiFolderId_1, fileExtension_1, message_1, ...args_1) => __awaiter(void 0, [documentId_1, documentTitle_1, companyId_1, createdBy_1, createdByName_1, createdByEmail_1, createdByPhotoURL_1, workspaceId_1, divisiFolderId_1, fileExtension_1, message_1, ...args_1], void 0, function* (documentId, documentTitle, companyId, createdBy, createdByName, createdByEmail, createdByPhotoURL, workspaceId, divisiFolderId, fileExtension, message, notificationType = "newDocumentNotification" // default to newDocumentNotification
) {
    try {
        // Fetch users in the company
        const usersSnapshot = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(companyId)
            .collection("users")
            .get();
        // Filter users based on the specific notification type in their settings
        const recipients = usersSnapshot.docs
            .map((doc) => {
            var _a, _b;
            const userData = doc.data();
            // Check if the specified notificationType is true in user settings
            if ((_b = (_a = userData === null || userData === void 0 ? void 0 : userData.settings) === null || _a === void 0 ? void 0 : _a.notifications) === null || _b === void 0 ? void 0 : _b[notificationType]) {
                return { id: doc.id, isRead: false };
            }
            return null;
        })
            .filter(Boolean); // Remove null values to keep only users with the notification enabled
        // If no recipients are found with the notification enabled, do nothing
        if (recipients.length === 0) {
            console.log(`No recipients found with ${notificationType} enabled.`);
            return;
        }
        // Generate a new document reference to get the notification ID
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc(); // Generate an empty document reference
        const notificationId = notificationRef.id; // Get the generated ID
        const notification = {
            id: notificationId, // Use the generated ID here
            documentId,
            documentTitle,
            createdBy,
            createdByName,
            createdByEmail,
            createdByPhotoURL,
            workspaceId,
            fileExtension,
            folderId: divisiFolderId, // Add divisiFolderId here
            recipients,
            message: message || `A new document "${documentTitle}" has been created.`,
            createdAt: new Date().toISOString(),
            comments: [],
        };
        yield notificationRef.set(notification);
        return notification;
    }
    catch (error) {
        console.error("Error sending notification:", error);
        throw error;
    }
});
exports.sendNotification = sendNotification;
const sendNotificationAccess = (documentId_1, documentTitle_1, fileExtension_1, userId_1, userName_1, userPhotoURL_1, companyId_1, workspaceId_1, divisiFolderId_1, ...args_1) => __awaiter(void 0, [documentId_1, documentTitle_1, fileExtension_1, userId_1, userName_1, userPhotoURL_1, companyId_1, workspaceId_1, divisiFolderId_1, ...args_1], void 0, function* (documentId, documentTitle, fileExtension, userId, userName, userPhotoURL, companyId, workspaceId, divisiFolderId, documentType = "document", message, invitedUsers) {
    try {
        if (invitedUsers.length === 0) {
            console.log("No invited users for this notification.");
            return null; // If no invited users, do nothing
        }
        // Generate a new document reference for the notification
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc();
        const notificationId = notificationRef.id; // Get the generated ID
        const notification = {
            id: notificationId,
            type: documentType,
            notificationsDetails: {
                documentId,
                documentTitle,
                fileExtension,
                folderId: divisiFolderId || null,
                message,
                createdAt: new Date().toISOString(),
                companyId,
                workspaceId,
            },
            user: {
                id: userId,
                name: userName,
                photoURL: userPhotoURL || null,
            },
            recipients: invitedUsers,
            comments: [],
        };
        // Save the notification in Firestore
        yield notificationRef.set(notification);
        return notification;
    }
    catch (error) {
        console.error("Error sending notification:", error);
        throw error;
    }
});
exports.sendNotificationAccess = sendNotificationAccess;
const sendNotificationRekamanAccess = (rekamanId_1, rekamanTitle_1, fileExtension_1, userId_1, userName_1, userPhotoURL_1, companyId_1, workspaceId_1, divisiFolderId_1, ...args_1) => __awaiter(void 0, [rekamanId_1, rekamanTitle_1, fileExtension_1, userId_1, userName_1, userPhotoURL_1, companyId_1, workspaceId_1, divisiFolderId_1, ...args_1], void 0, function* (rekamanId, rekamanTitle, fileExtension, userId, userName, userPhotoURL, companyId, workspaceId, divisiFolderId, documentType = "document", message, invitedUsers) {
    try {
        if (invitedUsers.length === 0) {
            console.log("No invited users for this notification.");
            return null; // If no invited users, do nothing
        }
        // Generate a new document reference for the notification
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc();
        const notificationId = notificationRef.id; // Get the generated ID
        const notification = {
            id: notificationId,
            type: documentType,
            notificationsDetails: {
                documentId: rekamanId,
                documentTitle: rekamanTitle,
                fileExtension,
                folderId: divisiFolderId || null,
                message,
                createdAt: new Date().toISOString(),
                companyId,
                workspaceId,
            },
            user: {
                id: userId,
                name: userName,
                photoURL: userPhotoURL || null,
            },
            recipients: invitedUsers,
            comments: [],
        };
        // Save the notification in Firestore
        yield notificationRef.set(notification);
        return notification;
    }
    catch (error) {
        console.error("Error sending notification:", error);
        throw error;
    }
});
exports.sendNotificationRekamanAccess = sendNotificationRekamanAccess;
const sendNotificationFolderAccess = (folderName_1, workspaceId_1, userId_1, userName_1, userPhotoURL_1, divisiFolderId_1, ...args_1) => __awaiter(void 0, [folderName_1, workspaceId_1, userId_1, userName_1, userPhotoURL_1, divisiFolderId_1, ...args_1], void 0, function* (folderName, workspaceId, userId, userName, userPhotoURL, divisiFolderId, documentType = "folder", message, invitedUsers = []) {
    try {
        if (invitedUsers.length === 0) {
            console.log("No invited users for this notification.");
            return null; // If no invited users, do nothing
        }
        // Generate a new document reference for the notification
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc();
        const notificationId = notificationRef.id; // Get the generated ID
        const notification = {
            id: notificationId,
            type: documentType,
            notificationsDetails: {
                folderName,
                folderId: divisiFolderId || null,
                message,
                createdAt: new Date().toISOString(),
            },
            recipients: invitedUsers.map((user) => ({
                id: user.id,
                email: user.email || "", // Fallback to empty string if null
                name: user.name || "", // Fallback to empty string if null
                photoURL: user.photoURL || null, // Fallback to null if undefined
                isRead: false, // Default to unread
            })),
            user: {
                id: userId,
                name: userName,
                photoURL: userPhotoURL || null,
            },
            comments: [],
        };
        // Save the notification in Firestore
        yield notificationRef.set(notification);
        return notification;
    }
    catch (error) {
        console.error("Error sending notification:", error);
        throw error;
    }
});
exports.sendNotificationFolderAccess = sendNotificationFolderAccess;
const sendNotificationSingle = (taskId_1, taskTitle_1, companyId_1, userId_1, userName_1, userPhotoURL_1, createdFor_1, createdForName_1, createdForEmail_1, createdForPhotoURL_1, workspaceId_1, type_1, message_1, ...args_1) => __awaiter(void 0, [taskId_1, taskTitle_1, companyId_1, userId_1, userName_1, userPhotoURL_1, createdFor_1, createdForName_1, createdForEmail_1, createdForPhotoURL_1, workspaceId_1, type_1, message_1, ...args_1], void 0, function* (taskId, taskTitle, companyId, userId, userName, userPhotoURL, createdFor, createdForName, createdForEmail, createdForPhotoURL, workspaceId, type, message, notificationType = "newDocumentNotification") {
    var _a, _b;
    try {
        // Fetch PIC user data from the company
        const picDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(companyId)
            .collection("users")
            .doc(createdFor)
            .get();
        if (!picDoc.exists) {
            console.log("PIC user not found for notification.");
            return;
        }
        const userData = picDoc.data();
        if (!((_b = (_a = userData === null || userData === void 0 ? void 0 : userData.settings) === null || _a === void 0 ? void 0 : _a.notifications) === null || _b === void 0 ? void 0 : _b[notificationType])) {
            console.log(`PIC has ${notificationType} notifications disabled.`);
            return;
        }
        // Generate a new document reference for the notification ID
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc();
        const notificationId = notificationRef.id;
        const notification = {
            id: notificationId,
            type,
            notificationsDetails: {
                taskTitle,
                taskId,
                message,
                createdAt: new Date().toISOString(),
            },
            recipients: [{
                    id: createdFor,
                    email: createdForEmail,
                    name: createdForName,
                    photoURL: createdForPhotoURL || null,
                    isRead: false,
                }],
            user: {
                id: userId,
                name: userName,
                photoURL: userPhotoURL || null,
            },
            comments: [],
        };
        yield notificationRef.set(notification);
        return notification;
    }
    catch (error) {
        console.error("Error sending notification to PIC:", error);
        throw error;
    }
});
exports.sendNotificationSingle = sendNotificationSingle;
const sendNotificationDocumentApproval = (documentApprovalId_1, documentId_1, documentTitle_1, companyId_1, userId_1, userName_1, userPhotoURL_1, recipients_1, workspaceId_1, isManual_1, type_1, message_1, ...args_1) => __awaiter(void 0, [documentApprovalId_1, documentId_1, documentTitle_1, companyId_1, userId_1, userName_1, userPhotoURL_1, recipients_1, workspaceId_1, isManual_1, type_1, message_1, ...args_1], void 0, function* (documentApprovalId, documentId, documentTitle, companyId, userId, userName, userPhotoURL, recipients, workspaceId, isManual, type, message, notificationType = "newDocumentNotification") {
    var _a, _b;
    try {
        // Check notification settings for each recipient
        const validRecipients = [];
        for (const recipient of recipients) {
            const picDoc = yield firebase_admin_1.adminDb
                .collection("companies")
                .doc(companyId)
                .collection("users")
                .doc(recipient.id)
                .get();
            if (!picDoc.exists) {
                console.log(`Recipient user ${recipient.id} not found for notification.`);
                continue;
            }
            const userData = picDoc.data();
            if (!((_b = (_a = userData === null || userData === void 0 ? void 0 : userData.settings) === null || _a === void 0 ? void 0 : _a.notifications) === null || _b === void 0 ? void 0 : _b[notificationType])) {
                console.log(`Recipient ${recipient.id} has ${notificationType} notifications disabled.`);
                continue;
            }
            validRecipients.push({
                id: recipient.id,
                email: recipient.email,
                name: recipient.name,
                photoURL: recipient.photoURL || null,
                isRead: false,
            });
        }
        if (validRecipients.length === 0) {
            console.log("No valid recipients for notification.");
            return;
        }
        // Generate a new document reference for the notification ID
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc();
        const notificationId = notificationRef.id;
        const notification = {
            id: notificationId,
            type,
            notificationsDetails: {
                documentApprovalId,
                isManual,
                documentTitle,
                documentId,
                message,
                createdAt: new Date().toISOString(),
            },
            recipients: validRecipients,
            user: {
                id: userId,
                name: userName,
                photoURL: userPhotoURL || null,
            },
            comments: [],
        };
        yield notificationRef.set(notification);
        return notification;
    }
    catch (error) {
        console.error("Error sending notification to recipients:", error);
        throw error;
    }
});
exports.sendNotificationDocumentApproval = sendNotificationDocumentApproval;
const updateRecipientIsRead = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, notificationId } = req.params;
        const user = req.user;
        // Fetch the specific notification
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc(notificationId);
        const notificationSnapshot = yield notificationRef.get();
        if (!notificationSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Notification not found.",
            });
        }
        const notificationData = notificationSnapshot.data();
        // Find if the recipient exists in the notification
        const recipientExists = notificationData === null || notificationData === void 0 ? void 0 : notificationData.recipients.some((recipient) => recipient.id === user.id);
        if (!recipientExists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Recipient not found in the notification.",
            });
        }
        // Update the specific recipient's isRead status to true
        const updatedRecipients = notificationData === null || notificationData === void 0 ? void 0 : notificationData.recipients.map((recipient) => {
            if (recipient.id === user.id) {
                return Object.assign(Object.assign({}, recipient), { isRead: true });
            }
            return recipient;
        });
        // Save the updated recipients array back to Firestore
        yield notificationRef.update({ recipients: updatedRecipients });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Notification marked as read.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.updateRecipientIsRead = updateRecipientIsRead;
const markAllNotificationsAsRead = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const user = req.user;
        // Fetch all notifications for the workspace
        const notificationsSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .get();
        // Check if there are no notifications
        if (notificationsSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No notifications found for this workspace.",
            });
        }
        let recipientFound = false;
        // Create a batch to update all notifications in one go
        const batch = firebase_admin_1.adminDb.batch();
        notificationsSnapshot.forEach((notificationDoc) => {
            const notificationData = notificationDoc.data();
            // Update the specific recipient's isRead status to true in each notification
            const updatedRecipients = notificationData.recipients.map((recipient) => {
                if (recipient.id === user.id) {
                    recipientFound = true;
                    return Object.assign(Object.assign({}, recipient), { isRead: true });
                }
                return recipient;
            });
            // Add the update operation to the batch
            const notificationRef = firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("notifications")
                .doc(notificationDoc.id);
            batch.update(notificationRef, { recipients: updatedRecipients });
        });
        // If recipient not found in any notification
        if (!recipientFound) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Recipient not found in any notifications.",
            });
        }
        // Commit the batch update
        yield batch.commit();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "All notifications marked as read for the recipient.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.markAllNotificationsAsRead = markAllNotificationsAsRead;
const viewNotifications = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const { limit, pageToken } = req.query; // Pagination params
        const user = req.user;
        // Set default limit if not provided
        const notificationsLimit = limit ? parseInt(limit, 10) : 10;
        // Ensure pageToken is a string or undefined
        const nextPageToken = pageToken ? pageToken : undefined;
        // Query for notifications in the workspace
        let notificationsQuery = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .orderBy("createdAt", "desc")
            .limit(notificationsLimit);
        // Pagination logic: start after the last notification
        if (nextPageToken) {
            const lastVisibleNotification = yield firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("notifications")
                .doc(nextPageToken)
                .get();
            if (lastVisibleNotification.exists) {
                notificationsQuery = notificationsQuery.startAfter(lastVisibleNotification);
            }
            else {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: "Invalid page token.",
                });
            }
        }
        // Fetch notifications
        const notificationsSnapshot = yield notificationsQuery.get();
        if (notificationsSnapshot.empty) {
            return res.status(200).json({
                status: "success",
                code: 200,
                data: [],
                message: "No notifications found.",
            });
        }
        // Filter notifications by whether the user is a recipient
        const notifications = notificationsSnapshot.docs
            .map((doc) => doc.data())
            .filter((notification) => notification.recipients.some((recipient) => recipient.id === user.id));
        // Get the last notification for pagination
        const lastNotification = notificationsSnapshot.docs[notificationsSnapshot.docs.length - 1];
        return res.status(200).json({
            status: "success",
            code: 200,
            data: notifications,
            nextPageToken: lastNotification ? lastNotification.id : null,
        });
    }
    catch (error) {
        console.error("Error fetching notifications:", error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewNotifications = viewNotifications;
