"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateCompanyDocuments = void 0;
const pizzip_1 = __importDefault(require("pizzip"));
const docxtemplater_1 = __importDefault(require("docxtemplater"));
const docxtemplater_image_module_free_1 = __importDefault(require("docxtemplater-image-module-free")); // Use the open-docxtemplater-image-module
const exceljs_1 = __importDefault(require("exceljs"));
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const image_size_1 = __importDefault(require("image-size")); // To get image dimensions
const firestore_1 = require("firebase-admin/firestore"); // For Firestore updates
const crypto_1 = __importDefault(require("crypto"));
const firebase_1 = require("../../../../utils/firebase");
const firestore_2 = require("firebase/firestore");
const sheetHandlers = new Map();
// Define handlers for each sheet with specific configurations
sheetHandlers.set("Cover", (workbook, imageId) => {
    const worksheet = workbook.getWorksheet("Cover");
    if (worksheet) {
        worksheet.addImage(imageId, {
            tl: { col: 1, row: 2 },
            br: { col: 4, row: 16 },
        });
    }
});
sheetHandlers.set("Aset Informasi", (workbook, imageId) => {
    const worksheet = workbook.getWorksheet("Aset Informasi");
    if (worksheet) {
        worksheet.addImage(imageId, {
            tl: { col: 0, row: 0 },
            br: { col: 2, row: 6 },
        });
    }
});
sheetHandlers.set("Daftar Kontak", (workbook, imageId) => {
    const worksheet = workbook.getWorksheet("Daftar Kontak");
    if (worksheet) {
        worksheet.addImage(imageId, {
            tl: { col: 1, row: 0 },
            br: { col: 3, row: 4 },
        });
    }
});
const formatDateIndonesian = (date) => {
    return new Intl.DateTimeFormat("id-ID", {
        day: "2-digit",
        month: "long",
        year: "numeric",
        hour: "2-digit",
        minute: "2-digit",
        hour12: true,
    }).format(date);
};
const wordExtensions = [
    "doc",
    "docm",
    "docx",
    "dot",
    "dotm",
    "dotx",
    "epub",
    "fb2",
    "fodt",
    "htm",
    "html",
    "mht",
    "mhtml",
    "odt",
    "ott",
    "rtf",
    "stw",
    "sxw",
    "txt",
    "wps",
    "wpt",
    "xml",
];
const cellExtensions = [
    "csv",
    "et",
    "ett",
    "fods",
    "ods",
    "ots",
    "sxc",
    "xls",
    "xlsb",
    "xlsm",
    "xlsx",
    "xlt",
    "xltm",
    "xltx",
    "xml",
];
const generateCompanyDocuments = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { workspaceId } = req.params;
        const { companyName } = req.body;
        const companyLogoFile = req.file; // Uploaded logo
        const user = req.user; // Assuming you are using authentication
        if (!companyName) {
            return res.status(400).json({ message: "Company name is required" });
        }
        if (!companyLogoFile || !companyLogoFile.buffer) {
            return res.status(400).json({ message: "Company logo is required" });
        }
        // Step 1: Fetch company data and current storage usage
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user.companyId);
        const companySnapshot = yield companyRef.get();
        const companyData = companySnapshot.data();
        if (!companyData) {
            return res.status(404).json({ message: "Company not found." });
        }
        const { storageLimit = 0 } = companyData.plan; // Storage limit in MB
        const { usage = {} } = companyData; // Usage field (could be empty initially)
        const currentStorageUsage = usage.storageUsage || 0; // Storage usage in MB
        // Step 2: Create a division folder for the company
        const folderName = `ISO 27001 Documents - ${companyName}`;
        const description = `Division folder for ${companyName}`;
        const divisiFolderId = yield createDivisiFolderHelper(folderName, description, user.companyId, user.id, workspaceId);
        if (!divisiFolderId) {
            return res.status(500).json({
                message: "Failed to create division folder.",
            });
        }
        const categoryRef = (0, firestore_2.collection)(firebase_1.db, "categories");
        let baseQuery = (0, firestore_2.query)(categoryRef, (0, firestore_2.and)((0, firestore_2.where)("workspaceId", "==", workspaceId), (0, firestore_2.or)((0, firestore_2.where)("categoryName", "==", "Prosedur"), (0, firestore_2.where)("categoryName", "==", "prosedur"))), (0, firestore_2.limit)(1));
        const existingCategorySnapshot = yield (0, firestore_2.getDocs)(baseQuery);
        let prosedurCategoryId;
        if (existingCategorySnapshot.empty) {
            const categoryRef = firebase_admin_1.adminDb.collection("categories").doc();
            yield categoryRef.set({
                id: categoryRef.id,
                categoryName: "Prosedur",
                workspaceId,
                companyId: user.companyId,
                createdAt: new Date().toISOString(),
                createdBy: user.id,
            });
            prosedurCategoryId = categoryRef.id;
        }
        else {
            prosedurCategoryId = existingCategorySnapshot.docs[0].id;
        }
        // Step 3: List all templates from the 'Template ISO 27001' folder in Firebase Storage
        const [files] = yield firebase_admin_1.bucket.getFiles({
            prefix: "Master/Template ISO 27001/",
        });
        // Ensure there are templates in the folder
        if (files.length === 0) {
            return res.status(404).json({ message: "No templates found." });
        }
        let totalGeneratedSize = 0; // Track the total size of generated documents
        for (const file of files) {
            console.log(file);
            const relativePath = file.name.replace("Master/Template ISO 27001/", "");
            const folderName = relativePath.split("/")[0];
            console.log(relativePath);
            if (!folderName || (folderName !== "Annex" && folderName !== "Klausul")) {
                continue; // Skip files not inside the specified folders
            }
            const [templateBuffer] = yield file.download();
            const fileExtension = (_a = file.name.split(".").pop()) === null || _a === void 0 ? void 0 : _a.toLowerCase();
            if (wordExtensions.includes(fileExtension || "")) {
                // DOCX handling with Docxtemplater
                const zip = new pizzip_1.default(templateBuffer);
                const imageOptions = {
                    getImage: (tagValue) => {
                        if (tagValue === "companyLogo") {
                            return companyLogoFile.buffer;
                        }
                        return null;
                    },
                    getSize: (img) => {
                        const dimensions = (0, image_size_1.default)(img);
                        const width = 150; // Default width for the logo
                        const height = (dimensions === null || dimensions === void 0 ? void 0 : dimensions.height) && (dimensions === null || dimensions === void 0 ? void 0 : dimensions.width)
                            ? Math.round((dimensions.height / dimensions.width) * width)
                            : 100; // Fallback height
                        return [width, height];
                    },
                    centered: true,
                    fileType: "docx",
                };
                const imageModule = new docxtemplater_image_module_free_1.default(imageOptions);
                const doc = new docxtemplater_1.default()
                    .attachModule(imageModule)
                    .loadZip(zip)
                    .setData({ companyLogo: "companyLogo", companyName })
                    .render();
                const generatedBuffer = doc.getZip().generate({ type: "nodebuffer" });
                const fileNameParts = file.name.split("/");
                const originalFileName = fileNameParts[fileNameParts.length - 1].replace(".docx", "");
                const outputFileName = `${originalFileName} - ${companyName}.docx`;
                const generatedDocSizeMB = generatedBuffer.length / (1024 * 1024);
                totalGeneratedSize += generatedDocSizeMB;
                if (currentStorageUsage + totalGeneratedSize > storageLimit) {
                    return res.status(400).json({
                        message: `Storage limit exceeded. Your plan allows ${storageLimit} MB, but current usage is ${currentStorageUsage} MB. Adding these documents would exceed the limit.`,
                    });
                }
                const outputFile = firebase_admin_1.bucket.file(`${user.companyId}/${companyName}/documents/ISO 27001/${outputFileName}`);
                yield outputFile.save(generatedBuffer, {
                    contentType: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
                });
                const documentRef = firebase_admin_1.adminDb.collection("documents").doc();
                const generateDocumentKey = () => {
                    return crypto_1.default.randomBytes(16).toString("hex"); // Example 16-byte random key
                };
                let newKey = generateDocumentKey();
                const historyEntry = {
                    created: formatDateIndonesian(new Date()),
                    key: newKey,
                    user: {
                        id: user.id,
                        name: user.name || "Unknown User",
                    },
                    historyData: {
                        fileType: "docx",
                        key: newKey,
                        url: `${user.companyId}/${companyName}/documents/ISO 27001/${outputFileName}`,
                    },
                    version: 1,
                };
                yield documentRef.set({
                    id: documentRef.id,
                    companyId: user.companyId,
                    title: originalFileName,
                    workspaceId,
                    storagePath: `${user.companyId}/${companyName}/documents/ISO 27001/${outputFileName}`,
                    divisiFolderId,
                    fileExtension: "docx",
                    createdAt: new Date().toISOString(),
                    createdBy: user.id,
                    createdByName: (user === null || user === void 0 ? void 0 : user.name) || "Unknown User",
                    createdByEmail: (user === null || user === void 0 ? void 0 : user.email) || "Unknown Email",
                    createdByPhotoURL: (user === null || user === void 0 ? void 0 : user.photoURL) || null,
                    categoryId: prosedurCategoryId,
                    categoryName: "Prosedur",
                    tags: "penting",
                    status: "Dalam Proses",
                    currentVersion: 1,
                    access: {
                        ownerId: user.id,
                        allowedRoles: ["admin", "superadmin"],
                        invitedUsers: [],
                    },
                    history: [historyEntry],
                    key: newKey,
                    generatedAt: firestore_1.FieldValue.serverTimestamp(),
                });
            }
            else if (cellExtensions.includes(fileExtension || "")) {
                // XLSX handling with ExcelJS
                const workbook = new exceljs_1.default.Workbook();
                yield workbook.xlsx.load(templateBuffer);
                if (companyLogoFile && companyLogoFile.buffer) {
                    const imageId = workbook.addImage({
                        buffer: companyLogoFile.buffer,
                        extension: "png",
                    });
                    sheetHandlers.forEach((handler, sheetName) => {
                        if (workbook.getWorksheet(sheetName)) {
                            handler(workbook, imageId);
                        }
                    });
                }
                const generatedBuffer = yield workbook.xlsx.writeBuffer();
                const fileNameParts = file.name.split("/");
                const originalFileName = fileNameParts[fileNameParts.length - 1].replace(".xlsx", "");
                const outputFileName = `${originalFileName} - ${companyName}.xlsx`;
                const generatedDocSizeMB = generatedBuffer.length / (1024 * 1024);
                totalGeneratedSize += generatedDocSizeMB;
                if (currentStorageUsage + totalGeneratedSize > storageLimit) {
                    return res.status(400).json({
                        message: `Storage limit exceeded. Your plan allows ${storageLimit} MB, but current usage is ${currentStorageUsage} MB. Adding these documents would exceed the limit.`,
                    });
                }
                const outputFile = firebase_admin_1.bucket.file(`${user.companyId}/${companyName}/documents/ISO 27001/${outputFileName}`);
                yield outputFile.save(generatedBuffer, {
                    contentType: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
                });
                const documentRef = firebase_admin_1.adminDb.collection("documents").doc();
                const generateDocumentKey = () => {
                    return crypto_1.default.randomBytes(16).toString("hex"); // Example 16-byte random key
                };
                let newKey = generateDocumentKey();
                const historyEntry = {
                    created: formatDateIndonesian(new Date()),
                    key: newKey,
                    user: {
                        id: user.id,
                        name: user.name || "Unknown User",
                    },
                    historyData: {
                        fileType: "xslx",
                        key: newKey,
                        url: `${user.companyId}/${companyName}/documents/ISO 27001/${outputFileName}`,
                    },
                    version: 1,
                };
                yield documentRef.set({
                    id: documentRef.id,
                    companyId: user.companyId,
                    title: originalFileName,
                    workspaceId,
                    storagePath: `${user.companyId}/${companyName}/documents/ISO 27001/${outputFileName}`,
                    divisiFolderId,
                    createdAt: new Date().toISOString(),
                    createdBy: user.id,
                    createdByName: (user === null || user === void 0 ? void 0 : user.name) || "Unknown User",
                    createdByEmail: (user === null || user === void 0 ? void 0 : user.email) || "Unknown Email",
                    createdByPhotoURL: (user === null || user === void 0 ? void 0 : user.photoURL) || null,
                    categoryId: prosedurCategoryId,
                    categoryName: "Prosedur",
                    tags: "penting",
                    status: "Dalam Proses",
                    fileExtension: "xlsx",
                    access: {
                        ownerId: user.id,
                        allowedRoles: ["admin", "superadmin"],
                        invitedUsers: [],
                    },
                    history: [historyEntry],
                    key: newKey,
                    generatedAt: firestore_1.FieldValue.serverTimestamp(),
                });
            }
        }
        // Update the company's storage usage in Firestore
        yield companyRef.update({
            "usage.storageUsage": firestore_1.FieldValue.increment(totalGeneratedSize),
        });
        return res.status(200).json({
            message: "Documents generated successfully",
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ message: "Error generating documents", error });
    }
});
exports.generateCompanyDocuments = generateCompanyDocuments;
// Helper function to create division folder
const createDivisiFolderHelper = (folderName, description, companyId, createdBy, workspaceId) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const folderRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("folders")
            .doc();
        const folderId = folderRef.id;
        yield folderRef.set({
            id: folderId,
            folderName,
            description,
            parentFolderId: null,
            workspaceId,
            companyId,
            createdBy,
            createdAt: new Date().toISOString(),
            access: {
                allowedRoles: ["superadmin", "admin"],
                assignedUsers: [],
                requiresApproval: true,
            },
        });
        return folderId;
    }
    catch (error) {
        console.error("Error creating division folder", error);
        return null;
    }
});
