"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDocumentFile = exports.deleteDocument = exports.viewDocument = exports.viewAllDocumentsInFolder = exports.viewAllDocuments = exports.editDocument = exports.createDocument = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const zod_1 = require("zod");
const document_1 = require("../../../../types/document");
const trash_1 = require("./trash");
const notifications_1 = require("./notifications");
const firestore_1 = require("firebase-admin/firestore");
const firestore_2 = require("firebase/firestore");
const crypto_1 = __importDefault(require("crypto"));
const firebase_1 = require("../../../../utils/firebase");
const createDocument = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { workspaceId } = req.params;
        const parsedBody = document_1.createDocumentSchema.parse(req.body);
        const { isManual, title, content, categoryId, status, tags, divisiFolderId, extension, } = parsedBody;
        const user = req.user;
        // Step 1: Validate workspace ownership
        const workspaceRef = firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId);
        const workspaceSnapshot = yield workspaceRef.get();
        if (!workspaceSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Workspace not found." });
        }
        const workspaceData = workspaceSnapshot.data();
        if (workspaceData.companyId !== user.companyId) {
            return res
                .status(403)
                .json({ status: "error", code: 403, message: "Access denied." });
        }
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user.companyId);
        const companySnapshot = yield companyRef.get();
        const companyData = companySnapshot.data();
        if (!companyData) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Company not found." });
        }
        const { storageLimit = 0 } = companyData.plan; // Storage limit in MB
        const { usage = {} } = companyData; // Usage field (could be empty initially)
        const currentStorageUsage = usage.storageUsage || 0; // Storage usage in MB
        // Step 3: Calculate the size of the manual document or uploaded file
        let newDocumentSize = 0; // Initialize size in MB
        let fileExtension = ""; // Store the file extension
        if (isManual) {
            // Manual document: Estimate size based on content and metadata
            const contentSize = Buffer.byteLength(content, "utf-8");
            const metadataSize = Buffer.byteLength(title, "utf-8") + JSON.stringify(tags).length;
            const firestoreOverhead = 1024; // 1 KB Firestore document overhead
            // Total size in MB
            newDocumentSize =
                (contentSize + metadataSize + firestoreOverhead) / (1024 * 1024);
            // Use the extension provided by the frontend via req.body
            fileExtension = extension || "url"; //
        }
        else {
            // File upload: Calculate size from the uploaded file (req.file)
            const docs = req.file;
            if (!docs) {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: "No file uploaded.",
                });
            }
            // Get file size in MB (docs.size is in bytes)
            newDocumentSize = docs.size / (1024 * 1024);
            // Extract file extension from the uploaded file name
            fileExtension = docs.originalname.split(".").pop() || "unknown";
        }
        // Step 4: Check if adding the new document exceeds the storage limit
        if (currentStorageUsage + newDocumentSize > storageLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Your plan allows ${storageLimit} MB, but the current usage is ${currentStorageUsage} MB. Adding this document would exceed the limit.`,
            });
        }
        // Step 5: Validate folder, category, and user access
        const folderRef = workspaceRef.collection("folders").doc(divisiFolderId);
        const folderSnapshot = yield folderRef.get();
        if (!folderSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Folder not found." });
        }
        const folderData = folderSnapshot.data();
        const hasAccess = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles.includes(user.role)) ||
            (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers.some((assignedUser) => assignedUser.id === user.id)) ||
            ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.ownerId) === user.id;
        if (!hasAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied to this folder.",
            });
        }
        const categoryRef = firebase_admin_1.adminDb.collection("categories").doc(categoryId);
        const categorySnapshot = yield categoryRef.get();
        if (!categorySnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Category not found." });
        }
        const categoryData = categorySnapshot.data();
        const userRef = firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users")
            .doc(user.id);
        const userSnapshot = yield userRef.get();
        const userData = userSnapshot.data();
        const formatDateIndonesian = (date) => {
            return new Intl.DateTimeFormat("id-ID", {
                day: "2-digit",
                month: "long",
                year: "numeric",
                hour: "2-digit",
                minute: "2-digit",
                hour12: true,
            }).format(date);
        };
        const generateDocumentKey = () => {
            return crypto_1.default.randomBytes(16).toString("hex"); // Example 16-byte random key
        };
        let newKey = generateDocumentKey();
        let documentData = {
            title,
            categoryId,
            categoryName: (categoryData === null || categoryData === void 0 ? void 0 : categoryData.categoryName) || "Uncategorized",
            divisiFolderId,
            folderName: (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || "Unknown Folder",
            workspaceId,
            status,
            tags,
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
            createdBy: user.id,
            createdByName: (userData === null || userData === void 0 ? void 0 : userData.name) || "Unknown User",
            createdByEmail: (userData === null || userData === void 0 ? void 0 : userData.email) || "Unknown Email",
            createdByPhotoURL: (userData === null || userData === void 0 ? void 0 : userData.photoURL) || null,
            access: {
                ownerId: user.id,
                allowedRoles: ["admin", "superadmin"],
                invitedUsers: [],
            },
            currentVersion: 1,
            fileExtension,
            key: newKey,
        };
        if (!isManual) {
            const sanitizedTitle = title
                .replace(/[^a-zA-Z0-9-_ ]/g, "")
                .replace(/ /g, "_");
            const newFileName = `${sanitizedTitle}.${fileExtension}`;
            const filePath = `${user.companyId}/workspaces/${workspaceId}/folders/${divisiFolderId}/documents/${newFileName}`;
            const storageFile = firebase_admin_1.bucket.file(filePath);
            yield storageFile.save(req.file.buffer, {
                metadata: { contentType: req.file.mimetype },
            });
            documentData.storagePath = filePath;
            const historyEntry = {
                created: formatDateIndonesian(new Date()),
                key: newKey,
                user: {
                    id: user.id,
                    name: user.name || "Unknown User",
                },
                historyData: {
                    fileType: fileExtension,
                    key: newKey,
                    url: filePath,
                },
                version: 1,
            };
            documentData.history = [historyEntry];
            documentData.key = newKey;
        }
        else {
            documentData.content = content; // Only for manual documents
        }
        // Step 7: Save document to Firestore
        const docRef = firebase_admin_1.adminDb.collection("documents").doc();
        yield docRef.set(Object.assign(Object.assign({}, documentData), { id: docRef.id }));
        // Step 8: Update the company's storage usage in Firestore
        yield companyRef.update({
            "usage.storageUsage": firestore_1.FieldValue.increment(newDocumentSize),
        });
        return res.status(201).json({
            status: "success",
            message: "Document created successfully.",
            data: documentData,
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.createDocument = createDocument;
const grantFolderAccessToUser = (workspaceRef_1, folderId_1, userId_1, email_1, name_1, role_1, photoURL_1, ...args_1) => __awaiter(void 0, [workspaceRef_1, folderId_1, userId_1, email_1, name_1, role_1, photoURL_1, ...args_1], void 0, function* (workspaceRef, folderId, userId, email, name, role, photoURL, folderType = "folders") {
    var _a;
    const userObj = {
        id: String(userId),
        email,
        name,
        role,
        photoURL,
    };
    while (folderId) {
        const folderRef = workspaceRef.collection(folderType).doc(folderId);
        const folderSnapshot = yield folderRef.get();
        const searchableDivisiFolderRef = firebase_admin_1.adminDb
            .collection("searchableDivisiFolder")
            .doc(folderId);
        const searchableDivisiFolderSnapshot = yield searchableDivisiFolderRef.get();
        if (!folderSnapshot.exists) {
            console.log(`Folder with ID ${folderId} not found.`);
            break;
        }
        const folderData = folderSnapshot.data();
        let assignedUsers = ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.assignedUsers) || [];
        // Check if user already exists in assignedUsers based on userId
        const userExists = assignedUsers.some((user) => user.userId === userObj.id);
        if (!userExists) {
            // Add the user object to access.assignedUsers and update the field
            yield folderRef.update({
                "access.assignedUsers": firestore_1.FieldValue.arrayUnion(userObj),
                "access.assignedUsersId": firestore_1.FieldValue.arrayUnion(userObj.id),
            });
            if (searchableDivisiFolderSnapshot.exists) {
                yield searchableDivisiFolderRef.update({
                    "access.assignedUsers": firestore_1.FieldValue.arrayUnion(userObj),
                    "access.assignedUsersId": firestore_1.FieldValue.arrayUnion(userObj.id),
                });
            }
            console.log(`User ${userObj.id} added to folder ${folderId}.`);
        }
        else {
            console.log(`User ${userObj.id} already exists in folder ${folderId}.`);
        }
        // Move to the next parent folder in the hierarchy
        folderId = folderData.parentFolderId || null;
    }
});
const editDocument = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f, _g, _h;
    try {
        const { documentId, workspaceId, divisiFolderId } = req.params;
        const parsedBody = document_1.editDocumentSchema.parse(req.body);
        const user = req.user;
        const workspaceRef = firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId);
        const documentRef = firebase_admin_1.adminDb.collection("documents").doc(documentId);
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user.companyId);
        const usersRef = companyRef.collection("users");
        const requests = [workspaceRef.get(), documentRef.get(), companyRef.get()];
        if (parsedBody.newDivisiFolderId) {
            const newFolderRef = workspaceRef
                .collection("folders")
                .doc(parsedBody.newDivisiFolderId);
            requests.push(newFolderRef.get());
        }
        if (parsedBody.categoryId) {
            const categoryRef = firebase_admin_1.adminDb
                .collection("categories")
                .doc(parsedBody.categoryId);
            requests.push(categoryRef.get());
        }
        const [workspaceSnapshot, documentSnapshot, companySnapshot] = yield Promise.all(requests);
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Workspace access denied.",
            });
        }
        if (!documentSnapshot.exists ||
            ((_b = documentSnapshot.data()) === null || _b === void 0 ? void 0 : _b.divisiFolderId) !== divisiFolderId) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Document not found." });
        }
        const documentData = documentSnapshot.data();
        const isOwnerOrAdmin = (documentData === null || documentData === void 0 ? void 0 : documentData.access.ownerId) === user.id ||
            ["admin", "superadmin"].includes(user.role);
        const isEditorInvitedUser = (_c = documentData === null || documentData === void 0 ? void 0 : documentData.access.invitedUsers) === null || _c === void 0 ? void 0 : _c.some((invitedUser) => invitedUser.id === user.id && invitedUser.role === "editor");
        if (!isOwnerOrAdmin && !isEditorInvitedUser) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to edit this document.",
            });
        }
        if (!companySnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Company not found." });
        }
        const companyData = companySnapshot.data();
        const existingInvitedUsers = ((_d = documentData === null || documentData === void 0 ? void 0 : documentData.access) === null || _d === void 0 ? void 0 : _d.invitedUsers) || [];
        const updatedInvitedUsers = [...existingInvitedUsers];
        const invitedUsers = parsedBody.invitedUsers || [];
        const fetchedUsers = [];
        for (const invitedUser of invitedUsers) {
            const companyUserSnapshot = yield usersRef.doc(invitedUser.id).get();
            if (!companyUserSnapshot.exists) {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: `User ID ${invitedUser.id} not found.`,
                });
            }
            const companyUser = companyUserSnapshot.data();
            const updatedUser = {
                id: companyUser.id,
                email: companyUser.email,
                name: companyUser.name,
                role: invitedUser.role,
                photoURL: (companyUser === null || companyUser === void 0 ? void 0 : companyUser.photoURL) || null,
            };
            const existingUserIndex = updatedInvitedUsers.findIndex((user) => user.id === invitedUser.id);
            if (existingUserIndex !== -1) {
                updatedInvitedUsers[existingUserIndex] = updatedUser;
            }
            else {
                updatedInvitedUsers.push(updatedUser);
            }
            fetchedUsers.push(updatedUser);
            yield grantFolderAccessToUser(workspaceRef, divisiFolderId, invitedUser.id, companyUser.email, companyUser.name, invitedUser.role, companyUser.photoURL || null);
        }
        if (invitedUsers.length > 0) {
            const documentType = "document";
            const formattedInvitedUsers = fetchedUsers.map((user) => ({
                id: user.id,
                email: user.email || "Unknown Email",
                name: user.name || "Unknown User",
                photoURL: user.photoURL || null,
                isRead: false,
            }));
            yield Promise.all(invitedUsers.map(() => (0, notifications_1.sendNotificationAccess)(documentId, documentData === null || documentData === void 0 ? void 0 : documentData.title, documentData === null || documentData === void 0 ? void 0 : documentData.fileExtension, user.id, user.name, (user === null || user === void 0 ? void 0 : user.photoURL) || undefined, user.companyId, workspaceId, divisiFolderId, documentType, `The ${documentType} "${documentData === null || documentData === void 0 ? void 0 : documentData.title}" has been shared with you by ${user.name}.`, formattedInvitedUsers)));
        }
        const { invitedUsers: invitedUsersFromBody, revokedUsers: revokedUsersFromBody } = parsedBody, restOfBody = __rest(parsedBody, ["invitedUsers", "revokedUsers"]);
        // Initialize revokedUsers as an empty array if not present
        const revokedUsers = revokedUsersFromBody || [];
        // Prepare an array to store the final invited users
        const revokedUserIds = revokedUsers.map((user) => user.id);
        const finalInvitedUsers = updatedInvitedUsers.filter((user) => !revokedUserIds.includes(user.id));
        const updatedDocumentData = Object.assign(Object.assign({}, restOfBody), { divisiFolderId: parsedBody.newDivisiFolderId || divisiFolderId, updatedAt: new Date().toISOString(), updatedBy: user.id, "access.invitedUsers": finalInvitedUsers });
        let newContentSize = 0;
        let oldContentSize = 0;
        if (parsedBody.content) {
            newContentSize = Buffer.byteLength(parsedBody.content, "utf-8");
        }
        if (documentData === null || documentData === void 0 ? void 0 : documentData.content) {
            oldContentSize = Buffer.byteLength(documentData.content, "utf-8");
        }
        const sizeDifference = (newContentSize - oldContentSize) / (1024 * 1024);
        if (sizeDifference > 0 &&
            (((_e = companyData === null || companyData === void 0 ? void 0 : companyData.usage) === null || _e === void 0 ? void 0 : _e.storageUsage) || 0) + sizeDifference >
                (((_f = companyData === null || companyData === void 0 ? void 0 : companyData.plan) === null || _f === void 0 ? void 0 : _f.storageLimit) || 0)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Your plan allows ${(_g = companyData === null || companyData === void 0 ? void 0 : companyData.plan) === null || _g === void 0 ? void 0 : _g.storageLimit} MB, but the current usage is ${(_h = companyData === null || companyData === void 0 ? void 0 : companyData.usage) === null || _h === void 0 ? void 0 : _h.storageUsage} MB.`,
            });
        }
        yield firebase_admin_1.adminDb.runTransaction((transaction) => __awaiter(void 0, void 0, void 0, function* () {
            transaction.update(documentRef, updatedDocumentData);
            if (sizeDifference !== 0) {
                transaction.update(companyRef, {
                    "usage.storageUsage": firestore_1.FieldValue.increment(sizeDifference),
                });
            }
        }));
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document updated successfully.",
        });
    }
    catch (error) {
        console.log(error.message);
        if (error instanceof zod_1.z.ZodError) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: error.errors[0],
            });
        }
        return res
            .status(500)
            .json({ status: "error", code: 500, message: error.message });
    }
});
exports.editDocument = editDocument;
const deleteDocument = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, divisiFolderId, documentId } = req.params;
        const documentRef = firebase_admin_1.adminDb.collection("documents").doc(documentId);
        const documentSnapshot = yield documentRef.get();
        const documentData = documentSnapshot.data();
        if (!documentSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document not found.",
            });
        }
        // Modify the req.body and req.params to fit what moveToTrash expects
        req.body = { folderId: divisiFolderId, documentId };
        req.params = { workspaceId };
        // Call moveToTrash directly
        yield (0, trash_1.moveToTrash)(req, res);
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteDocument = deleteDocument;
const getDocumentFile = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, documentId, divisiFolderId } = req.params; // Added divisiFolderId in the request params
        const user = req.user;
        // Step 1: Validate workspace ownership
        const workspaceRef = firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId);
        const workspaceSnapshot = yield workspaceRef.get();
        if (!workspaceSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Workspace not found." });
        }
        const workspaceData = workspaceSnapshot.data();
        if (workspaceData.companyId !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied to this workspace.",
            });
        }
        // Step 2: Check folder existence and user access
        const folderRef = workspaceRef.collection("folders").doc(divisiFolderId);
        const folderSnapshot = yield folderRef.get();
        if (!folderSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Folder not found." });
        }
        const folderData = folderSnapshot.data();
        const hasFolderAccess = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles.includes(user.role)) ||
            (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers.includes(user.id));
        if (!hasFolderAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied to this folder.",
            });
        }
        // Step 3: Fetch document details
        const docRef = firebase_admin_1.adminDb.collection("documents").doc(documentId); // Fetching from documents collection
        const docSnapshot = yield docRef.get();
        if (!docSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Document not found." });
        }
        const documentData = docSnapshot.data();
        // Step 4: Verify document access permissions (superadmin, admin, or invitedUsers)
        const hasDocumentAccess = (documentData === null || documentData === void 0 ? void 0 : documentData.access.allowedRoles.includes(user.role)) ||
            (documentData === null || documentData === void 0 ? void 0 : documentData.access.invitedUsers.includes(user.id));
        if (!hasDocumentAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied to this document.",
            });
        }
        // Step 5: Check if filePath is available and generate a signed URL for file access
        if (!(documentData === null || documentData === void 0 ? void 0 : documentData.filePath)) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "File not found",
            });
        }
        const file = firebase_admin_1.bucket.file(documentData.filePath);
        const [signedUrl] = yield file.getSignedUrl({
            action: "read",
            expires: Date.now() + 60 * 60 * 1000, // 1 hour expiration
        });
        return res.status(200).json({
            status: "success",
            message: "File access granted.",
            fileUrl: signedUrl,
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal server error." });
    }
});
exports.getDocumentFile = getDocumentFile;
const viewDocument = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { documentId, workspaceId, divisiFolderId } = req.params;
        const user = req.user;
        // Step 1: Fetch workspace, folder, and document data in parallel
        const [workspaceSnapshot, folderSnapshot, documentSnapshot] = yield Promise.all([
            firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId).get(),
            firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("folders")
                .doc(divisiFolderId)
                .get(),
            firebase_admin_1.adminDb.collection("documents").doc(documentId).get(),
        ]);
        // Verify workspace and access by company ID
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found or access denied.",
            });
        }
        // Folder access check
        if (!folderSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Folder not found." });
        }
        const folderData = folderSnapshot.data();
        const folderAllowedRoles = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles) || [];
        const folderAssignedUsers = (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers) || [];
        const folderRequiresApproval = folderData === null || folderData === void 0 ? void 0 : folderData.access.requiresApproval;
        const hasFolderAccess = folderAllowedRoles.includes(user.role) ||
            folderAssignedUsers.some((assignedUser) => assignedUser.id === user.id);
        if (!hasFolderAccess &&
            folderRequiresApproval &&
            user.role !== "superadmin" &&
            user.role !== "admin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access to this folder is denied.",
            });
        }
        // Check document access
        if (!documentSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Document not found." });
        }
        const documentData = documentSnapshot.data();
        const documentAllowedRoles = (documentData === null || documentData === void 0 ? void 0 : documentData.access.allowedRoles) || [];
        const documentInvitedUsers = (documentData === null || documentData === void 0 ? void 0 : documentData.access.invitedUsers) || [];
        const invitedUser = documentInvitedUsers.find((invited) => invited.id === user.id);
        const isViewerOnly = (invitedUser === null || invitedUser === void 0 ? void 0 : invitedUser.role) === "viewer";
        const hasDocumentAccess = documentAllowedRoles.includes(user.role) || invitedUser;
        if (!hasDocumentAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to view this document.",
            });
        }
        // Fetch category data
        const categorySnapshot = yield firebase_admin_1.adminDb
            .collection("categories")
            .doc(documentData.categoryId)
            .get();
        const categoryName = categorySnapshot.exists
            ? ((_b = categorySnapshot.data()) === null || _b === void 0 ? void 0 : _b.categoryName) || "Uncategorized"
            : "Uncategorized";
        // Prepare the document data for response
        const documentCopy = Object.assign({}, documentData);
        delete documentCopy.divisiFolderId;
        delete documentCopy.categoryId;
        // Prepare response data
        const responseData = Object.assign(Object.assign({}, documentCopy), { folder: {
                id: divisiFolderId,
                folderName: (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || "Unknown Folder",
            }, category: { id: documentData === null || documentData === void 0 ? void 0 : documentData.categoryId, categoryName } });
        // Only add `isViewerOnly` to response if true
        if (isViewerOnly) {
            responseData.isViewerOnly = true;
        }
        // Respond with document, folder, and category data
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document retrieved successfully.",
            data: responseData,
        });
    }
    catch (error) {
        console.error("Error viewing document:", error);
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.viewDocument = viewDocument;
const viewAllDocuments = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const user = req.user;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to view documents.",
            });
        }
        // Verify workspace ownership
        const workspaceDoc = yield (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "workspaces", workspaceId));
        if (!workspaceDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied.",
            });
        }
        const documentsRef = (0, firestore_2.collection)(firebase_1.db, "documents");
        const perPage = parseInt(req.query.perPage) || 10;
        const startAfterDocId = req.query.startAfter;
        const searchTerm = (req.query.q || "").toLowerCase();
        const userData = {
            email: user.email,
            id: user.id,
            name: user.name,
            photoURL: user.photoURL || null,
        };
        let baseQuery = (0, firestore_2.query)(documentsRef, (0, firestore_2.and)((0, firestore_2.where)("workspaceId", "==", workspaceId), (0, firestore_2.or)((0, firestore_2.where)("access.ownerId", "==", user.id), (0, firestore_2.where)("access.allowedRoles", "array-contains", user.role), (0, firestore_2.where)("access.invitedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "editor" })), (0, firestore_2.where)("access.invitedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "viewer" })))), (0, firestore_2.orderBy)("createdAt"), (0, firestore_2.limit)(perPage));
        if (searchTerm) {
            baseQuery = (0, firestore_2.query)(baseQuery, (0, firestore_2.where)("title", ">=", searchTerm), (0, firestore_2.where)("title", "<=", searchTerm + "\uf8ff"));
        }
        if (startAfterDocId) {
            const startAfterDoc = yield (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "documents", startAfterDocId));
            if (startAfterDoc.exists()) {
                baseQuery = (0, firestore_2.query)(baseQuery, (0, firestore_2.startAfter)(startAfterDoc));
            }
            else {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: `Document with startAfterDocId ${startAfterDocId} does not exist.`,
                });
            }
        }
        const documentsSnapshot = yield (0, firestore_2.getDocs)(baseQuery);
        if (documentsSnapshot.empty) {
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "No documents found in this workspace.",
                data: [],
                lastVisible: null,
            });
        }
        // Fetch all unique user IDs from documents for fetching user details
        const userIds = new Set(documentsSnapshot.docs.map((doc) => doc.data().createdBy));
        const userDocs = yield Promise.all(Array.from(userIds).map((id) => (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "users", id))));
        // Map user details
        const users = userDocs.reduce((acc, userDoc) => {
            if (userDoc.exists()) {
                acc[userDoc.id] = userDoc.data();
            }
            return acc;
        }, {});
        // Construct documents array with deduplication
        const documents = documentsSnapshot.docs.map((doc) => {
            const data = doc.data();
            return Object.assign({ id: doc.id }, data);
        });
        const lastVisible = documentsSnapshot.docs[documentsSnapshot.docs.length - 1];
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Documents fetched successfully.",
            data: documents,
            lastVisible: lastVisible ? lastVisible.id : null,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewAllDocuments = viewAllDocuments;
const viewAllDocumentsInFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { workspaceId, divisiFolderId } = req.params;
        const user = req.user;
        const [workspaceSnapshot, folderSnapshot] = yield Promise.all([
            firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId).get(),
            firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("folders")
                .doc(divisiFolderId)
                .get(),
        ]);
        if (!workspaceSnapshot.exists ||
            ((_a = workspaceSnapshot.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found or access denied.",
            });
        }
        if (!folderSnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Folder not found." });
        }
        const folderData = folderSnapshot.data();
        const hasFolderAccess = (folderData === null || folderData === void 0 ? void 0 : folderData.access.allowedRoles.includes(user.role)) ||
            (folderData === null || folderData === void 0 ? void 0 : folderData.access.assignedUsers.includes(user.id));
        if (!hasFolderAccess) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access to this folder is denied.",
            });
        }
        const documentsSnapshot = yield firebase_admin_1.adminDb
            .collection("documents")
            .where("workspaceId", "==", workspaceId)
            .where("divisiFolderId", "==", divisiFolderId)
            .get();
        const accessibleDocuments = documentsSnapshot.docs
            .map((doc) => doc.data())
            .filter((document) => {
            var _a;
            const { access } = document;
            const isOwner = (access === null || access === void 0 ? void 0 : access.ownerId) === (user === null || user === void 0 ? void 0 : user.id);
            const isAdminOrSuperAdmin = ["admin", "superadmin"].includes(user === null || user === void 0 ? void 0 : user.role);
            const isInvited = (_a = access === null || access === void 0 ? void 0 : access.invitedUsers) === null || _a === void 0 ? void 0 : _a.includes(user === null || user === void 0 ? void 0 : user.id);
            return isOwner || isAdminOrSuperAdmin || isInvited;
        });
        const categoryIds = Array.from(new Set(accessibleDocuments.map((doc) => doc.categoryId)));
        const categorySnapshots = yield Promise.all(categoryIds.map((id) => firebase_admin_1.adminDb.collection("categories").doc(id).get()));
        const categories = categorySnapshots.reduce((acc, snap) => {
            var _a;
            if (snap.exists) {
                acc[snap.id] = (_a = snap.data()) === null || _a === void 0 ? void 0 : _a.categoryName;
            }
            return acc;
        }, {});
        // Fetch user data from the company's users subcollection
        const userIds = Array.from(new Set(accessibleDocuments.map((doc) => doc.createdBy)));
        const userSnapshots = yield Promise.all(userIds.map((id) => firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId) // User's company ID
            .collection("users")
            .doc(id)
            .get()));
        const users = userSnapshots.reduce((acc, snap) => {
            var _a, _b, _c;
            if (snap.exists) {
                acc[snap.id] = {
                    name: (_a = snap.data()) === null || _a === void 0 ? void 0 : _a.name,
                    email: (_b = snap.data()) === null || _b === void 0 ? void 0 : _b.email,
                    photoURL: (_c = snap.data()) === null || _c === void 0 ? void 0 : _c.photoURL,
                };
            }
            return acc;
        }, {});
        const documentsWithFoldersCategoriesAndCreator = accessibleDocuments.map((doc) => {
            var _a, _b, _c;
            const documentCopy = Object.assign({}, doc);
            delete documentCopy.divisiFolderId;
            delete documentCopy.categoryId;
            return Object.assign(Object.assign({}, documentCopy), { folder: {
                    id: doc.divisiFolderId,
                    folderName: (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || "Unknown Folder",
                }, category: {
                    id: doc.categoryId,
                    categoryName: categories[doc.categoryId] || "Uncategorized",
                }, createdBy: {
                    id: doc.createdBy,
                    name: ((_a = users[doc.createdBy]) === null || _a === void 0 ? void 0 : _a.name) || "Unknown User",
                    email: ((_b = users[doc.createdBy]) === null || _b === void 0 ? void 0 : _b.email) || "Unknown Email",
                    photoURL: ((_c = users[doc.createdBy]) === null || _c === void 0 ? void 0 : _c.email) || null,
                } });
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Documents fetched successfully.",
            data: documentsWithFoldersCategoriesAndCreator,
        });
    }
    catch (error) {
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.viewAllDocumentsInFolder = viewAllDocumentsInFolder;
