"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.migrateAllFoldersToSearchableDivisiFolder = exports.deleteFolder = exports.viewFolderWithSubfolders = exports.viewFolder = exports.viewAllDivisiFolders = exports.editDivisiFolder = exports.createDivisiFolder = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const firebase_1 = require("../../../../utils/firebase");
const trash_1 = require("./trash");
const divisiFolder_1 = require("../../../../types/divisiFolder");
const zod_1 = require("zod");
const firestore_1 = require("firebase-admin/firestore");
const firestore_2 = require("firebase/firestore");
const typesense_1 = __importDefault(require("../../../../utils/typesense"));
const notifications_1 = require("./notifications");
const createDivisiFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const parsedBody = divisiFolder_1.createFolderSchema.parse(req.body);
        const { folderName, description, invitedUsers = [], parentFolderId = null, } = parsedBody;
        const user = req.user;
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to perform this action.",
            });
        }
        // Verify if the workspace exists
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You are not allowed to create folders in this workspace.",
            });
        }
        const companyDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .get();
        const companyData = companyDoc.data();
        if (!companyData) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Company not found.",
            });
        }
        const { storageLimit = 0 } = companyData.plan;
        const { usage = {} } = companyData;
        const currentStorageUsage = usage.storageUsage || 0;
        // Estimate folder size (name + description overhead) in bytes
        const folderSizeBytes = Buffer.byteLength(folderName, "utf-8") +
            Buffer.byteLength(description || "", "utf-8");
        // Convert bytes to MB
        const folderSizeMB = folderSizeBytes / (1024 * 1024);
        // Check if the folder creation exceeds storage limit
        if (currentStorageUsage + folderSizeMB > storageLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Allowed: ${storageLimit} MB, Used: ${currentStorageUsage} MB.`,
            });
        }
        // Validate parent folder existence if provided
        const folderCollectionRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("folders");
        if (parentFolderId) {
            const parentFolderDoc = yield folderCollectionRef
                .doc(parentFolderId)
                .get();
            if (!parentFolderDoc.exists) {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: "Parent folder not found.",
                });
            }
        }
        // Ensure no duplicate folder name under the same parent
        const existingFolderSnapshot = yield folderCollectionRef
            .where("workspaceId", "==", workspaceId)
            .where("folderName", "==", folderName)
            .where("parentFolderId", "==", parentFolderId)
            .limit(1)
            .get();
        if (!existingFolderSnapshot.empty) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "A folder with the same name already exists.",
            });
        }
        // Validate and process invited users with roles
        const validInvitedUsers = [];
        if (invitedUsers.length > 0) {
            const userDocs = yield firebase_admin_1.adminDb.getAll(...invitedUsers.map((user) => {
                var _a;
                return firebase_admin_1.adminDb
                    .collection("companies")
                    .doc((_a = req === null || req === void 0 ? void 0 : req.user) === null || _a === void 0 ? void 0 : _a.companyId)
                    .collection("users")
                    .doc(user.id);
            }));
            for (let i = 0; i < userDocs.length; i++) {
                const userDoc = userDocs[i];
                const invitedUser = invitedUsers[i];
                if (!userDoc.exists) {
                    return res.status(400).json({
                        status: "error",
                        code: 400,
                        message: `User with ID ${invitedUser.id} does not exist within this company.`,
                    });
                }
                const invitedUserData = userDoc.data();
                if ((invitedUserData === null || invitedUserData === void 0 ? void 0 : invitedUserData.companyId) !== user.companyId) {
                    return res.status(400).json({
                        status: "error",
                        code: 400,
                        message: `User with ID ${invitedUser.id} is not in the same company.`,
                    });
                }
                validInvitedUsers.push({
                    id: invitedUser.id,
                    email: (invitedUserData === null || invitedUserData === void 0 ? void 0 : invitedUserData.email) || null,
                    name: (invitedUserData === null || invitedUserData === void 0 ? void 0 : invitedUserData.name) || null,
                    photoURL: (invitedUserData === null || invitedUserData === void 0 ? void 0 : invitedUserData.photoURL) || null,
                    role: invitedUser.role,
                });
            }
        }
        // Create folder
        const folderRef = folderCollectionRef.doc();
        const folderId = folderRef.id;
        const folderNameInsensitive = folderName.toLowerCase();
        const folderData = {
            id: folderId,
            folderName,
            folderName_insensitive: folderNameInsensitive,
            description,
            parentFolderId: parentFolderId || null,
            workspaceId,
            createdAt: new Date().toISOString(),
            createdBy: user.id,
            access: {
                allowedRoles: ["superadmin", "admin"],
                assignedUsers: validInvitedUsers,
                assignedUsersId: validInvitedUsers.map((user) => user.id),
                requiresApproval: true,
                ownerId: user.id,
            },
        };
        yield folderRef.set(folderData);
        const searchDivisiFolderRef = firebase_admin_1.adminDb
            .collection("searchableDivisiFolder")
            .doc(folderId);
        yield searchDivisiFolderRef.set(Object.assign(Object.assign({}, folderData), { companyId: user.companyId }));
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .update({
            "usage.storageUsage": firestore_1.FieldValue.increment(folderSizeMB),
        });
        const notificationRecipients = validInvitedUsers.map((user) => ({
            id: user.id,
            email: (user === null || user === void 0 ? void 0 : user.email) || "",
            name: (user === null || user === void 0 ? void 0 : user.name) || "",
            photoURL: user.photoURL || undefined,
            isRead: false, // Default to unread
        }));
        if (validInvitedUsers.length > 0) {
            yield (0, notifications_1.sendNotificationFolderAccess)(folderName, workspaceId, user.id, user.name, (user === null || user === void 0 ? void 0 : user.photoURL) || undefined, folderId, "folder", `You have been granted access to the folder "${folderName}".`, notificationRecipients);
        }
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "Folder created successfully.",
            invitedUsers: validInvitedUsers,
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error.",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.createDivisiFolder = createDivisiFolder;
const migrateAllFoldersToSearchableDivisiFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Get all workspaces
        const workspaceSnapshot = yield firebase_admin_1.adminDb.collection("workspaces").get();
        if (workspaceSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No workspaces found to migrate folders from.",
            });
        }
        const migratedFolders = []; // To track migrated folder ids
        const updatedWorkspaces = []; // Track workspaces processed
        // Loop through each workspace
        for (const workspaceDoc of workspaceSnapshot.docs) {
            const workspaceId = workspaceDoc.id;
            const workspaceData = workspaceDoc.data();
            // Get all folders from the current workspace
            const folderCollectionRef = firebase_admin_1.adminDb
                .collection("workspaces")
                .doc(workspaceId)
                .collection("folders");
            const folderSnapshot = yield folderCollectionRef.get();
            if (folderSnapshot.empty) {
                continue; // Skip if no folders exist in this workspace
            }
            // Loop through each folder in the workspace and migrate to searchableDivisiFolder
            for (const folderDoc of folderSnapshot.docs) {
                const folderData = folderDoc.data();
                const folderId = folderDoc.id;
                const folderNameInsensitive = folderData.folderName.toLowerCase();
                const folderDataForMigration = {
                    id: folderId,
                    folderName: folderData.folderName,
                    folderName_insensitive: folderNameInsensitive,
                    description: folderData.description || "",
                    parentFolderId: folderData.parentFolderId || null,
                    workspaceId,
                    createdAt: folderData.createdAt,
                    createdBy: folderData.createdBy,
                    access: folderData.access || {
                        allowedRoles: ["superadmin", "admin"],
                        assignedUsers: [],
                        requiresApproval: true,
                        ownerId: folderData.createdBy,
                    },
                    companyId: workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId,
                };
                // Migrate folder to searchableDivisiFolder collection
                const searchDivisiFolderRef = firebase_admin_1.adminDb
                    .collection("searchableDivisiFolder")
                    .doc(folderId);
                yield searchDivisiFolderRef.set(folderDataForMigration);
                // Track the migrated folder
                migratedFolders.push(folderId);
                // Optionally update the storage usage if necessary (assuming no new content is being added here)
                const folderSizeMB = Buffer.byteLength(folderData.folderName, "utf-8") / (1024 * 1024); // Estimate size in MB
                yield firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId)
                    .update({
                    "usage.storageUsage": firestore_1.FieldValue.increment(folderSizeMB),
                });
            }
            // Track workspace as processed
            updatedWorkspaces.push(workspaceId);
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            message: `${migratedFolders.length} folder(s) migrated from ${updatedWorkspaces.length} workspace(s).`,
            migratedFolders,
            updatedWorkspaces,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.migrateAllFoldersToSearchableDivisiFolder = migrateAllFoldersToSearchableDivisiFolder;
function mergeAssignedUsers(currentAssignedUsers, newInvitedUsers, explicitRevokedUsers, companyId) {
    return __awaiter(this, void 0, void 0, function* () {
        const userMap = {};
        // Fetch user data from Firestore for new invited users
        const userPromises = newInvitedUsers.map((user) => __awaiter(this, void 0, void 0, function* () {
            const userDoc = yield firebase_admin_1.adminDb
                .collection("companies")
                .doc(companyId)
                .collection("users")
                .doc(user.id)
                .get();
            if (userDoc.exists) {
                const userData = userDoc.data();
                return {
                    id: user.id,
                    email: (userData === null || userData === void 0 ? void 0 : userData.email) || null,
                    name: (userData === null || userData === void 0 ? void 0 : userData.name) || null,
                    photoURL: (userData === null || userData === void 0 ? void 0 : userData.photoURL) || null,
                    role: user.role,
                };
            }
            return null;
        }));
        // Await and filter non-null fetched users
        const fetchedInvitedUsers = (yield Promise.all(userPromises)).filter(Boolean);
        fetchedInvitedUsers.forEach((user) => {
            userMap[user === null || user === void 0 ? void 0 : user.id] = user;
        });
        // Add current assigned users to the map
        currentAssignedUsers.forEach((user) => {
            const isRevoked = explicitRevokedUsers.some((revokedUser) => revokedUser.id === user.id);
            if (!isRevoked) {
                const invitedUserWithNewRole = newInvitedUsers.find((invitedUser) => invitedUser.id === user.id);
                if (invitedUserWithNewRole)
                    user.role = invitedUserWithNewRole.role;
                userMap[user.id] = Object.assign(Object.assign({}, userMap[user.id]), user);
            }
        });
        return Object.values(userMap);
    });
}
// const grantFolderAccess = async (
//   workspaceRef: FirebaseFirestore.DocumentReference,
//   folderId: string,
//   userId: string,
//   email: string,
//   name: string,
//   role: string,
//   photoURL: string,
//   folderType: string = "folders"
// ) => {
//   const userObj = { id: userId, email, name, role, photoURL };
//   while (folderId) {
//     const folderRef = workspaceRef.collection(folderType).doc(folderId);
//     const folderSnapshot = await folderRef.get();
//     const searchableDivisiFolderRef = adminDb
//       .collection("searchableDivisiFolder")
//       .doc(folderId);
//     const searchableDivisiFolderSnapshot =
//       await searchableDivisiFolderRef.get();
//     if (!folderSnapshot.exists) {
//       console.log(`Folder with ID ${folderId} not found.`);
//       break;
//     }
//     const folderData = folderSnapshot.data();
//     const assignedUsers = folderData?.access?.assignedUsers || [];
//     const userExists = assignedUsers.some(
//       (user: any) => user.id === userObj.id
//     );
//     if (!userExists) {
//       await folderRef.update({
//         "access.assignedUsers": FieldValue.arrayUnion(userObj),
//         "access.assignedUsersId": FieldValue.arrayUnion(userObj.id),
//       });
//       if (searchableDivisiFolderSnapshot.exists) {
//         await searchableDivisiFolderRef.update({
//           "access.assignedUsers": FieldValue.arrayUnion(userObj),
//           "access.assignedUsersId": FieldValue.arrayUnion(userObj.id),
//         });
//       }
//       console.log(`User ${userObj.id} added to folder ${folderId}.`);
//     } else {
//       console.log(
//         `User ${userObj.id} already has access to folder ${folderId}.`
//       );
//     }
//     folderId = folderData?.parentFolderId || null;
//   }
// };
const grantFolderAccess = (workspaceRef_1, userId_1, userName_1, userPhotoURL_1, companyId_1, folderId_1, invitedUsers_1, ...args_1) => __awaiter(void 0, [workspaceRef_1, userId_1, userName_1, userPhotoURL_1, companyId_1, folderId_1, invitedUsers_1, ...args_1], void 0, function* (workspaceRef, userId, userName, userPhotoURL, companyId, folderId, invitedUsers, // Only user IDs are passed
folderType = "folders") {
    var _a;
    const addedUsers = [];
    // Fetch full details for each invited user
    const userDetailsPromises = invitedUsers.map((user) => __awaiter(void 0, void 0, void 0, function* () {
        const userDoc = yield firebase_admin_1.adminDb.collection("companies").doc(companyId).collection("users").doc(user.id).get();
        if (!userDoc.exists) {
            console.log(`User with ID ${user.id} not found.`);
            return null;
        }
        const userData = userDoc.data();
        return {
            id: user.id,
            email: (userData === null || userData === void 0 ? void 0 : userData.email) || "",
            name: (userData === null || userData === void 0 ? void 0 : userData.name) || "",
            photoURL: (userData === null || userData === void 0 ? void 0 : userData.photoURL) || null,
        };
    }));
    const resolvedUserDetails = (yield Promise.all(userDetailsPromises)).filter((user) => user !== null);
    while (folderId) {
        const folderRef = workspaceRef.collection(folderType).doc(folderId);
        const folderSnapshot = yield folderRef.get();
        if (!folderSnapshot.exists) {
            console.log(`Folder with ID ${folderId} not found.`);
            break;
        }
        const folderData = folderSnapshot.data();
        const assignedUsers = ((_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.assignedUsers) || [];
        for (const userObj of resolvedUserDetails) {
            const userExists = assignedUsers.some((user) => user.id === userObj.id);
            if (!userExists) {
                // Update Firestore with new user access
                yield folderRef.update({
                    "access.assignedUsers": firestore_1.FieldValue.arrayUnion(userObj),
                    "access.assignedUsersId": firestore_1.FieldValue.arrayUnion(userObj.id),
                });
                console.log(`User ${userObj.id} added to folder ${folderId}.`);
                addedUsers.push({
                    id: userObj.id,
                    email: userObj.email,
                    name: userObj.name,
                    photoURL: (userObj === null || userObj === void 0 ? void 0 : userObj.photoURL) || undefined,
                    isRead: false,
                });
            }
            else {
                console.log(`User ${userObj.id} already has access to folder ${folderId}.`);
            }
        }
        // Send notification if this is the top-level parent folder
        if (!(folderData === null || folderData === void 0 ? void 0 : folderData.parentFolderId) && addedUsers.length > 0) {
            yield (0, notifications_1.sendNotificationFolderAccess)(folderData.folderName, workspaceRef.id, userId, userName, userPhotoURL, folderId, "folder", `You have been granted access to the folder "${folderData.folderName}".`, addedUsers);
            console.log(`Notification sent for folder ${folderId} as it is the top-level folder.`);
            // Clear addedUsers after sending notification
            addedUsers.length = 0;
        }
        folderId = (folderData === null || folderData === void 0 ? void 0 : folderData.parentFolderId) || null;
    }
});
const editDivisiFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d;
    try {
        const { workspaceId, folderId } = req.params;
        const user = req.user;
        if (!workspaceId || !folderId || !(user === null || user === void 0 ? void 0 : user.companyId)) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID, Folder ID, and user authorization are required.",
            });
        }
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user.companyId);
        const companySnapshot = yield companyRef.get();
        if (!companySnapshot.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Company not found." });
        }
        const parsedBody = divisiFolder_1.editFolderSchema.parse(req.body);
        const { folderName, description, invitedUsers, revokedUsers, newParentFolderId, } = parsedBody;
        const [workspaceDoc, folderDoc] = yield firebase_admin_1.adminDb.getAll(firebase_admin_1.adminDb.collection("workspaces").doc(workspaceId), firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("folders")
            .doc(folderId));
        if (!workspaceDoc.exists ||
            ((_a = workspaceDoc.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You do not have permission to edit folders in this workspace.",
            });
        }
        if (!folderDoc.exists) {
            return res
                .status(404)
                .json({ status: "error", code: 404, message: "Folder not found." });
        }
        const folderData = folderDoc.data();
        const isAdminOrEditor = ["superadmin", "admin"].includes(user === null || user === void 0 ? void 0 : user.role) ||
            ((_c = (_b = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _b === void 0 ? void 0 : _b.assignedUsers) === null || _c === void 0 ? void 0 : _c.some((u) => u.id === user.id && u.role === "editor"));
        const isFolderCreator = (folderData === null || folderData === void 0 ? void 0 : folderData.createdBy) === user.id;
        if (!isAdminOrEditor && !isFolderCreator) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You are not allowed to edit this folder.",
            });
        }
        // Calculate current folder size
        const currentFolderSize = Buffer.byteLength(folderData === null || folderData === void 0 ? void 0 : folderData.folderName, "utf-8") +
            Buffer.byteLength((folderData === null || folderData === void 0 ? void 0 : folderData.description) || "", "utf-8");
        // Calculate new folder size
        const newFolderSize = Buffer.byteLength(folderName || (folderData === null || folderData === void 0 ? void 0 : folderData.folderName), "utf-8") +
            Buffer.byteLength(description || (folderData === null || folderData === void 0 ? void 0 : folderData.description) || "", "utf-8");
        const sizeDifference = newFolderSize - currentFolderSize;
        const companyDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .get();
        const companyData = companyDoc.data();
        if (!companyData) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Company not found.",
            });
        }
        const { storageLimit = 0 } = companyData.plan; // Storage limit in MB
        const { usage = {} } = companyData; // Usage field (could be empty initially)
        const currentStorageUsage = usage.storageUsage || 0; // Storage usage in MB
        // Calculate potential new storage usage
        const potentialStorageUsage = currentStorageUsage + sizeDifference / (1024 * 1024); // Convert bytes to MB
        // Check if the potential new usage exceeds the storage limit
        if (potentialStorageUsage > storageLimit) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: `Storage limit exceeded. Allowed: ${storageLimit} MB, Used: ${currentStorageUsage} MB.`,
            });
        }
        const updates = {};
        if (folderName) {
            updates.folderName = folderName;
            updates.folderName_insensitive = folderName.toLowerCase();
        }
        if (description)
            updates.description = description;
        if (isAdminOrEditor && (invitedUsers || revokedUsers)) {
            const updatedAssignedUsers = yield mergeAssignedUsers(((_d = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _d === void 0 ? void 0 : _d.assignedUsers) || [], invitedUsers || [], revokedUsers || [], user.companyId);
            updates["access.assignedUsers"] = updatedAssignedUsers;
            updates["access.assignedUsersId"] = updatedAssignedUsers.map((user) => user.id);
            if (invitedUsers && invitedUsers.length > 0) {
                yield grantFolderAccess(workspaceDoc.ref, user.id, user.name, user.photoURL || undefined, user.companyId, folderId, invitedUsers, "folders");
            }
        }
        if (newParentFolderId !== undefined) {
            const newParentFolderDoc = newParentFolderId
                ? yield firebase_admin_1.adminDb
                    .collection("workspaces")
                    .doc(workspaceId)
                    .collection("folders")
                    .doc(newParentFolderId)
                    .get()
                : null;
            if (newParentFolderDoc === null || newParentFolderDoc === void 0 ? void 0 : newParentFolderDoc.exists) {
                updates.parentFolderId = newParentFolderId;
            }
            else {
                updates.parentFolderId = null;
            }
        }
        if (Object.keys(updates).length === 0) {
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "No changes were made.",
            });
        }
        updates.updatedAt = new Date().toISOString();
        updates.updatedBy = user.id;
        yield folderDoc.ref.update(updates);
        const searchDivisiFolderRef = firebase_admin_1.adminDb
            .collection("searchableDivisiFolder")
            .doc(folderId);
        const folderDataForSearch = Object.assign(Object.assign(Object.assign({}, folderData), updates), { companyId: user.companyId });
        yield searchDivisiFolderRef.set(folderDataForSearch);
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .update({
            "usage.storageUsage": firestore_1.FieldValue.increment(sizeDifference / (1024 * 1024)), // Increment storage usage
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Folder updated successfully.",
        });
    }
    catch (error) {
        console.log(error);
        if (error instanceof zod_1.z.ZodError) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: error.errors[0],
            });
        }
        return res
            .status(500)
            .json({ status: "error", code: 500, message: "Internal Server Error" });
    }
});
exports.editDivisiFolder = editDivisiFolder;
const viewAllDivisiFolders = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const user = req.user;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to view folders.",
            });
        }
        const workspaceDoc = yield (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "workspaces", workspaceId));
        if (!workspaceDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied.",
            });
        }
        const foldersRef = (0, firestore_2.collection)(firebase_1.db, "workspaces", workspaceId, "folders");
        const perPage = parseInt(req.query.perPage) || 10;
        const startAfterDocId = req.query.startAfter;
        const searchTerm = (req.query.q || "").toLowerCase();
        const page = parseInt(req.query.page) || 1;
        const userData = {
            email: user.email,
            id: user.id,
            name: user.name,
            photoURL: user.photoURL || null,
        };
        let countQuery = (0, firestore_2.query)(foldersRef, (0, firestore_2.and)((0, firestore_2.where)("parentFolderId", "==", null), (0, firestore_2.or)((0, firestore_2.where)("access.ownerId", "==", user.id), (0, firestore_2.where)("access.allowedRoles", "array-contains", user.role), (0, firestore_2.where)("access.assignedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "editor" })), (0, firestore_2.where)("access.assignedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "viewer" })))));
        let baseQuery = (0, firestore_2.query)(foldersRef, (0, firestore_2.and)((0, firestore_2.where)("parentFolderId", "==", null), (0, firestore_2.or)((0, firestore_2.where)("access.ownerId", "==", user.id), (0, firestore_2.where)("access.allowedRoles", "array-contains", user.role), (0, firestore_2.where)("access.assignedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "editor" })), (0, firestore_2.where)("access.assignedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "viewer" })))), (0, firestore_2.orderBy)(searchTerm ? "folderName_insensitive" : "createdAt"), (0, firestore_2.limit)(perPage));
        let totalDivisiFolders = 0;
        let folderIdsFromSearch = [];
        if (searchTerm) {
            try {
                const searchParameters = {
                    q: searchTerm,
                    query_by: "folderName_insensitive",
                    filter_by: `workspaceId:${workspaceId} && companyId:${user.companyId} && (
             access.allowedRoles:=${user.role} || access.assignedUsersId:=${user.id} 
          )`
                        .replace(/\s+/g, " ")
                        .trim(),
                    per_page: perPage,
                    page,
                };
                const searchResults = yield typesense_1.default
                    .collections("searchableDivisiFolder")
                    .documents()
                    .search(searchParameters);
                folderIdsFromSearch = searchResults === null || searchResults === void 0 ? void 0 : searchResults.hits.map((hit) => { var _a; return (_a = hit.document) === null || _a === void 0 ? void 0 : _a.id; });
                totalDivisiFolders = searchResults.found;
            }
            catch (error) {
                console.log(error);
            }
        }
        if (searchTerm && folderIdsFromSearch.length === 0) {
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "No folders found matching the search criteria.",
                data: [],
                totalDivisiFolders: 0,
                lastVisible: null,
            });
        }
        if (folderIdsFromSearch.length > 0) {
            baseQuery = (0, firestore_2.query)(foldersRef, (0, firestore_2.where)("id", "in", folderIdsFromSearch));
            countQuery = (0, firestore_2.query)(foldersRef, (0, firestore_2.where)("id", "in", folderIdsFromSearch));
        }
        if (!searchTerm) {
            const totalFoldersSnapshot = yield (0, firestore_2.getDocs)(countQuery);
            totalDivisiFolders = totalFoldersSnapshot.size;
        }
        if (!searchTerm && startAfterDocId) {
            const startAfterDoc = yield (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "workspaces", workspaceId, "folders", startAfterDocId));
            if (startAfterDoc.exists()) {
                baseQuery = (0, firestore_2.query)(baseQuery, (0, firestore_2.startAfter)(startAfterDoc));
            }
            else {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: `Document with startAfterDocId ${startAfterDocId} does not exist.`,
                });
            }
        }
        const parentFoldersSnapshot = yield (0, firestore_2.getDocs)(baseQuery);
        if (parentFoldersSnapshot.empty) {
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "No folders found in this workspace.",
                data: [],
                totalDivisiFolders: 0,
                lastVisible: null,
            });
        }
        const fetchDocumentCounts = (folderId) => __awaiter(void 0, void 0, void 0, function* () {
            const documentSnapshot = yield (0, firestore_2.getDocs)((0, firestore_2.query)((0, firestore_2.collection)(firebase_1.db, "documents"), (0, firestore_2.and)((0, firestore_2.where)("divisiFolderId", "==", folderId), (0, firestore_2.or)((0, firestore_2.where)("access.ownerId", "==", user.id), (0, firestore_2.where)("access.allowedRoles", "array-contains", user.role), (0, firestore_2.where)("access.invitedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "editor" })), (0, firestore_2.where)("access.invitedUsers", "array-contains", Object.assign(Object.assign({}, userData), { role: "viewer" }))))));
            return documentSnapshot.size;
        });
        const fetchSubfoldersWithCounts = (parentFolderId) => __awaiter(void 0, void 0, void 0, function* () {
            const subfoldersSnapshot = yield (0, firestore_2.getDocs)((0, firestore_2.query)(foldersRef, (0, firestore_2.where)("parentFolderId", "==", parentFolderId)));
            const subfolders = yield Promise.all(subfoldersSnapshot.docs.map((subfolderDoc) => __awaiter(void 0, void 0, void 0, function* () {
                const data = subfolderDoc.data();
                const documentCount = yield fetchDocumentCounts(subfolderDoc.id);
                const nestedSubfolders = yield fetchSubfoldersWithCounts(subfolderDoc.id);
                const totalDocumentCount = documentCount +
                    nestedSubfolders.reduce((sum, sub) => sum + sub.documentCount, 0);
                return Object.assign(Object.assign({ id: subfolderDoc.id }, data), { documentCount: totalDocumentCount, subfolders: nestedSubfolders });
            })));
            return subfolders;
        });
        const parentFolders = yield Promise.all(parentFoldersSnapshot.docs.map((folderDoc) => __awaiter(void 0, void 0, void 0, function* () {
            const data = folderDoc.data();
            const documentCount = yield fetchDocumentCounts(folderDoc.id);
            const subfolders = yield fetchSubfoldersWithCounts(folderDoc.id);
            const totalDocumentCount = documentCount +
                subfolders.reduce((sum, sub) => sum + sub.documentCount, 0);
            const createdByUserDoc = yield (0, firestore_2.getDoc)((0, firestore_2.doc)(firebase_1.db, "companies", user.companyId, "users", data.createdBy));
            const createdByUser = createdByUserDoc.exists()
                ? createdByUserDoc.data()
                : { name: "Unknown User", email: "No Email", photoURL: null };
            return Object.assign(Object.assign({ id: folderDoc.id }, data), { createdByUser, documentCount: totalDocumentCount, subfolders });
        })));
        const lastVisible = parentFoldersSnapshot.docs[parentFoldersSnapshot.docs.length - 1];
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Folders fetched successfully.",
            data: parentFolders,
            totalDivisiFolders,
            lastVisible: lastVisible ? lastVisible.id : null,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewAllDivisiFolders = viewAllDivisiFolders;
const viewFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { workspaceId, folderId } = req.params;
        const user = req.user;
        if (!workspaceId || !folderId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID and Folder ID are required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to perform this action.",
            });
        }
        // Check if the workspace exists and belongs to the user's company
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        const workspaceData = workspaceDoc.data();
        if ((workspaceData === null || workspaceData === void 0 ? void 0 : workspaceData.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You do not have permission to view folders in this workspace.",
            });
        }
        // Fetch the specific folder
        const folderDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("folders")
            .doc(folderId)
            .get();
        if (!folderDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Folder not found.",
            });
        }
        const folderData = folderDoc.data();
        // Check access permissions based on roles
        const isAdminOrSuperadmin = (user === null || user === void 0 ? void 0 : user.role) === "superadmin" || (user === null || user === void 0 ? void 0 : user.role) === "admin";
        const isUserAssigned = (_b = (_a = folderData === null || folderData === void 0 ? void 0 : folderData.access) === null || _a === void 0 ? void 0 : _a.assignedUsers) === null || _b === void 0 ? void 0 : _b.some((assignedUser) => assignedUser.id === user.id);
        if (!isAdminOrSuperadmin && !isUserAssigned) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Access denied. You are not allowed to view this folder.",
            });
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Folder fetched successfully.",
            data: folderData,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewFolder = viewFolder;
const viewFolderWithSubfolders = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, folderId } = req.params;
        // Fetch the parent folder (main folder)
        const folderDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("folders")
            .doc(folderId)
            .get();
        if (!folderDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Folder not found.",
            });
        }
        const folderData = folderDoc.data();
        // Fetch subfolders once
        const subfoldersSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("folders")
            .where("parentFolderId", "==", folderId)
            .get();
        // Get the parent folder name once
        const parentFolderName = (folderData === null || folderData === void 0 ? void 0 : folderData.folderName) || null;
        // Map subfolders and reuse the parentFolderName
        const subfolders = subfoldersSnapshot.docs.map((doc) => (Object.assign({ id: doc.id, parentFolderName }, doc.data())));
        return res.status(200).json({
            status: "success",
            data: {
                folder: Object.assign({ id: folderDoc.id }, folderData),
                subfolders,
            },
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewFolderWithSubfolders = viewFolderWithSubfolders;
const deleteFolder = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        req.body = { folderId: req.params.folderId };
        return yield (0, trash_1.moveToTrash)(req, res);
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteFolder = deleteFolder;
