"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteComment = exports.updateComment = exports.createComment = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const uuid_1 = require("uuid");
const firestore_1 = require("firebase-admin/firestore");
const createComment = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, notificationId } = req.params;
        const user = req.user;
        const { content } = req.body;
        if (!workspaceId || !notificationId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "workspaceId and notificationId are required.",
            });
        }
        if (!content) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Content is required.",
            });
        }
        const commentId = (0, uuid_1.v4)();
        const comment = {
            id: commentId,
            content,
            createdBy: user.id,
            createdByName: user.name || "Unknown User",
            createdAt: new Date().toISOString(),
            createdByPhotoURL: user.photoURL,
        };
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc(notificationId);
        const notificationDoc = yield notificationRef.get();
        if (!notificationDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Notification not found.",
            });
        }
        // Add the comment to the notification's `comments` array
        yield notificationRef.update({
            comments: firestore_1.FieldValue.arrayUnion(comment),
        });
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "Comment added successfully.",
            data: comment,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.createComment = createComment;
const updateComment = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, notificationId, commentId } = req.params;
        const user = req.user;
        const { content } = req.body;
        // Validate required fields
        if (!workspaceId || !notificationId || !commentId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "workspaceId, notificationId, and commentId are required.",
            });
        }
        if (!content) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Content is required to update the comment.",
            });
        }
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc(notificationId);
        const notificationDoc = yield notificationRef.get();
        const notificationData = notificationDoc.data();
        if (!notificationData) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Notification not found.",
            });
        }
        const comments = notificationData.comments || [];
        const commentIndex = comments.findIndex((c) => c.id === commentId);
        if (commentIndex === -1) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Comment not found.",
            });
        }
        const comment = comments[commentIndex];
        if (comment.createdBy !== user.id) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You can only edit your own comments.",
            });
        }
        comments[commentIndex] = Object.assign(Object.assign({}, comment), { content, updatedAt: new Date().toISOString() });
        yield notificationRef.update({ comments });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Comment updated successfully.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.updateComment = updateComment;
const deleteComment = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, notificationId, commentId } = req.params;
        const user = req.user;
        // Validate required fields
        if (!workspaceId || !notificationId || !commentId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "workspaceId, notificationId, and commentId are required.",
            });
        }
        const notificationRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("notifications")
            .doc(notificationId);
        const notificationDoc = yield notificationRef.get();
        const notificationData = notificationDoc.data();
        if (!notificationData) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Notification not found.",
            });
        }
        const comments = notificationData.comments || [];
        const commentIndex = comments.findIndex((c) => c.id === commentId);
        if (commentIndex === -1) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Comment not found.",
            });
        }
        const comment = comments[commentIndex];
        if (comment.createdBy !== user.id) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You can only delete your own comments.",
            });
        }
        comments.splice(commentIndex, 1);
        yield notificationRef.update({ comments });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Comment deleted successfully.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteComment = deleteComment;
