"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.changePassword = exports.updateUserProfile = exports.logout = exports.checkAuth = exports.login = exports.resetPassword = exports.register = void 0;
const zod_1 = require("zod");
const users_1 = require("../../../../types/users");
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const auth_1 = require("firebase/auth");
const firebase_1 = require("../../../../utils/firebase");
const auth_2 = require("../../../../types/auth");
const auth_3 = require("firebase-admin/auth");
const authAdmin = (0, auth_3.getAuth)();
const updateUserReferences = (userId, updatedData) => __awaiter(void 0, void 0, void 0, function* () {
    // Update user's name, email, and photoURL in the `documents` collection
    const docsQuerySnapshot = yield firebase_admin_1.adminDb
        .collection("documents")
        .where("createdBy", "==", userId)
        .get();
    const batch = firebase_admin_1.adminDb.batch();
    docsQuerySnapshot.forEach((doc) => {
        batch.update(doc.ref, {
            createdByName: updatedData.name || doc.data().createdByName,
            createdByEmail: updatedData.email || doc.data().createdByEmail,
            createdByPhotoURL: updatedData.photoURL || doc.data().createdByPhotoURL, // Update photoURL
        });
    });
    // Update user's name, email, and photoURL in the `categories` collection
    const categoriesQuerySnapshot = yield firebase_admin_1.adminDb
        .collection("categories")
        .where("createdBy", "==", userId)
        .get();
    categoriesQuerySnapshot.forEach((categoryDoc) => {
        batch.update(categoryDoc.ref, {
            updatedByName: updatedData.name || categoryDoc.data().updatedByName,
            updatedByEmail: updatedData.email || categoryDoc.data().updatedByEmail,
            updatedByPhotoURL: updatedData.photoURL || categoryDoc.data().updatedByPhotoURL, // Update photoURL
        });
    });
    // Update user's name, email, and photoURL in the `rekaman` collection
    const rekamanQuerySnapshot = yield firebase_admin_1.adminDb
        .collection("rekaman")
        .where("createdBy", "==", userId)
        .get();
    rekamanQuerySnapshot.forEach((rekamanDoc) => {
        batch.update(rekamanDoc.ref, {
            createdByName: updatedData.name || rekamanDoc.data().createdByName,
            createdByEmail: updatedData.email || rekamanDoc.data().createdByEmail,
            createdByPhotoURL: updatedData.photoURL || rekamanDoc.data().createdByPhotoURL, // Update photoURL
        });
    });
    // Commit the batch update
    yield batch.commit();
});
const updateUserProfile = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const parsedBody = auth_2.userUpdateSchema.parse(req.body);
        const { name, email, password, phoneNumber, newDocumentNotification, commentNotification, approvalNotification, taskNotification, } = parsedBody;
        const currentUser = req.user;
        if (!currentUser) {
            return res.status(401).json({
                status: "error",
                code: 401,
                message: "User not authenticated.",
            });
        }
        // Fetch user details using Admin SDK
        const firebaseUser = yield authAdmin.getUser(currentUser.uid);
        let emailUpdated = false;
        let photoURL = null;
        // Prepare updates for Firebase Auth
        const updates = {};
        if (email && email !== firebaseUser.email) {
            updates.email = email;
            emailUpdated = true;
        }
        if (password) {
            updates.password = password;
        }
        if (name) {
            updates.displayName = name;
        }
        // Upload new photo if provided
        if (req.file) {
            const allowedExtensions = ["jpg", "jpeg", "png"];
            const maxFileSize = 5 * 1024 * 1024; // 5 MB in bytes
            const fileExtension = (_a = req.file.originalname
                .split(".")
                .pop()) === null || _a === void 0 ? void 0 : _a.toLowerCase();
            // Check file type
            if (!fileExtension || !allowedExtensions.includes(fileExtension)) {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: "Invalid file type. Only JPG and PNG are allowed.",
                });
            }
            // Check file size
            if (req.file.size > maxFileSize) {
                return res.status(400).json({
                    status: "error",
                    code: 400,
                    message: "File size exceeds the 5 MB limit.",
                });
            }
            const filePath = `${currentUser.companyId}/profile_photos/${currentUser.uid}.${fileExtension}`;
            const fileRef = firebase_admin_1.bucket.file(filePath);
            try {
                yield fileRef.save(req.file.buffer, {
                    metadata: {
                        contentType: req.file.mimetype,
                    },
                });
                yield fileRef.makePublic();
                photoURL = `https://storage.googleapis.com/${fileRef.bucket.name}/${fileRef.name}`;
                updates.photoURL = photoURL;
            }
            catch (fileError) {
                return res.status(500).json({
                    status: "error",
                    code: 500,
                    message: "Failed to upload profile picture.",
                    error: fileError.message,
                });
            }
        }
        // Apply updates to Firebase Auth user
        yield authAdmin.updateUser(currentUser.uid, updates);
        // Update Firestore user data
        const userDocRef = firebase_admin_1.adminDb
            .collection("companies")
            .doc(currentUser.companyId)
            .collection("users")
            .doc(currentUser.uid);
        const firestoreUpdates = {
            settings: {
                notifications: {},
            },
        };
        if (name !== undefined)
            firestoreUpdates.name = name;
        if (email !== undefined)
            firestoreUpdates.email = email;
        if (phoneNumber !== undefined)
            firestoreUpdates.phoneNumber = phoneNumber;
        if (photoURL)
            firestoreUpdates.photoURL = photoURL;
        if (newDocumentNotification !== undefined) {
            firestoreUpdates.settings.notifications.newDocumentNotification =
                newDocumentNotification;
        }
        if (commentNotification !== undefined) {
            firestoreUpdates.settings.notifications.commentNotification =
                commentNotification;
        }
        if (approvalNotification !== undefined) {
            firestoreUpdates.settings.notifications.approvalNotification =
                approvalNotification;
        }
        if (taskNotification !== undefined) {
            firestoreUpdates.settings.notifications.taskNotification =
                taskNotification;
        }
        yield userDocRef.update(firestoreUpdates);
        // Trigger updates in all references to this user
        try {
            yield updateUserReferences(currentUser.uid, {
                name: name || currentUser.name,
                email: email || currentUser.email,
                photoURL: photoURL || currentUser.photoURL,
            });
        }
        catch (refError) {
            return res.status(500).json({
                status: "error",
                code: 500,
                message: "Failed to update user references.",
                error: refError.message,
            });
        }
        return res.status(200).json({
            status: "success",
            message: emailUpdated
                ? "User updated successfully. Please verify your new email."
                : "User updated successfully.",
            data: {
                name,
                email,
                photoURL,
                phoneNumber,
                newDocumentNotification,
                commentNotification,
                approvalNotification,
                taskNotification,
            },
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: error.errors[0],
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: error.message || "Internal Server Error",
        });
    }
});
exports.updateUserProfile = updateUserProfile;
const changePassword = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Parse and validate the request body using Zod schema
        const parsedBody = users_1.changePasswordSchema.parse(req.body);
        const { currentPassword, password } = parsedBody;
        const currentUser = req.user;
        if (!currentUser || !currentUser.id) {
            return res.status(401).json({
                status: "error",
                code: 401,
                message: "User not authenticated.",
            });
        }
        // Fetch the Firebase user using Admin SDK
        const firebaseUser = yield authAdmin.getUser(currentUser.id);
        if (!firebaseUser.email) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "User email not found.",
            });
        }
        // Manually re-authenticate: Verify the current password
        const credential = auth_1.EmailAuthProvider.credential(firebaseUser.email, currentPassword);
        // Simulate re-authentication by signing in with the provided credential
        try {
            const userCredential = yield (0, auth_1.signInWithCredential)(firebase_1.auth, credential);
            if (!userCredential) {
                return res.status(401).json({
                    status: "error",
                    code: 401,
                    message: "Current password is incorrect.",
                });
            }
        }
        catch (error) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Current password is incorrect.",
            });
        }
        // Update the user's password directly using the Admin SDK
        yield authAdmin.updateUser(currentUser.id, { password });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Password changed successfully.",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: error.errors[0],
            });
        }
        // General error handler
        return res.status(500).json({
            status: "error",
            code: 500,
            message: error.message || "Internal Server Error",
        });
    }
});
exports.changePassword = changePassword;
const register = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const parsedBody = users_1.userRegisterSchema.parse(req.body);
        const { name, email, password, companyName, planId, roleId } = parsedBody;
        let companyRef = yield firebase_admin_1.adminDb
            .collection("companies")
            .where("name", "==", companyName)
            .get();
        let companyId;
        if (companyRef.empty) {
            const newCompanyRef = firebase_admin_1.adminDb.collection("companies").doc();
            const planRef = firebase_admin_1.adminDb.collection("plans").doc(planId || "Free");
            const planData = (yield planRef.get()).data();
            if (!planData) {
                return res.status(400).json({
                    status: "error",
                    message: "Invalid plan selected",
                });
            }
            yield newCompanyRef.set({
                name: companyName,
                plan: planData,
            });
            companyId = newCompanyRef.id;
        }
        else {
            companyId = companyRef.docs[0].id;
            if (planId) {
                const planRef = firebase_admin_1.adminDb.collection("plans").doc(planId);
                const planData = (yield planRef.get()).data();
                if (!planData) {
                    return res.status(400).json({
                        status: "error",
                        message: "Invalid plan selected",
                    });
                }
                yield firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(companyId)
                    .update({ plan: planData });
            }
        }
        const roleRef = firebase_admin_1.adminDb.collection("roles").doc(roleId || "viewer");
        const roleData = (yield roleRef.get()).data();
        if (!roleData) {
            return res.status(400).json({
                status: "error",
                message: "Invalid role selected",
            });
        }
        // Step 4: Register the user in Firebase Authentication
        const userCredential = yield (0, auth_1.createUserWithEmailAndPassword)(firebase_1.auth, email, password);
        const user = userCredential.user;
        // Step 5: Send email verification
        yield (0, auth_1.sendEmailVerification)(user);
        yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(companyId)
            .collection("users")
            .doc(user.uid)
            .set({
            id: user.uid,
            name,
            email,
            roleId: roleData.id,
            companyId,
            createdAt: new Date().toISOString(),
        });
        // Step 7: Return success response
        return res.status(201).json({
            status: "success",
            message: "Verification email sent! User created successfully!",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                message: "Validation error",
                error: firstError,
            });
        }
        // General error handler
        return res.status(500).json({
            status: "error",
            message: error.message || "Internal Server Error",
        });
    }
});
exports.register = register;
const login = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const parsedBody = users_1.userLoginSchema.parse(req.body);
        const { email, password } = parsedBody;
        // Sign in with email and password
        const userCredential = yield (0, auth_1.signInWithEmailAndPassword)(firebase_1.auth, email, password);
        const user = userCredential.user;
        // Check email verification
        if (!user.emailVerified) {
            return res.status(403).json({
                error: "Please verify your email before logging in",
            });
        }
        // Generate ID token and refresh token
        const idToken = yield user.getIdToken(true);
        const refreshToken = user.refreshToken; // Firebase automatically provides this
        // Set access and refresh tokens as HttpOnly cookies
        res.cookie("access_token", idToken, {
            httpOnly: true,
            secure: process.env.NODE_ENV === "production",
            sameSite: process.env.NODE_ENV === "production" ? "none" : "lax",
        });
        res.cookie("refresh_token", refreshToken, {
            httpOnly: true,
            secure: process.env.NODE_ENV === "production",
            sameSite: process.env.NODE_ENV === "production" ? "none" : "lax",
        });
        // Respond with success
        return res.status(200).json({
            message: "User logged in successfully",
            user: {
                uid: user.uid,
                email: user.email,
            },
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            message: error.message || "An error occurred while logging in",
        });
    }
});
exports.login = login;
const logout = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        yield (0, auth_1.signOut)(firebase_1.auth);
        res.clearCookie("access_token");
        res.clearCookie("refresh_token");
        res.status(200).json({ message: "User logged out successfully" });
    }
    catch (error) {
        res.status(500).json({ error: "Internal Server Error" });
    }
});
exports.logout = logout;
const resetPassword = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Validate email input (You can add Zod validation here if necessary)
        const { email } = req.body;
        if (!email) {
            return res.status(400).json({
                status: "error",
                message: "Email is required",
            });
        }
        yield (0, auth_1.sendPasswordResetEmail)(firebase_1.auth, email);
        // Return success response
        return res.status(200).json({
            status: "success",
            message: "Password reset email sent!",
        });
    }
    catch (error) {
        // Handle Firebase errors (e.g., invalid email)
        return res.status(500).json({
            status: "error",
            message: error.message || "An error occurred while sending reset email",
        });
    }
});
exports.resetPassword = resetPassword;
const checkAuth = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        // Reference to the company document
        const companyRef = firebase_admin_1.adminDb.collection("companies").doc(user === null || user === void 0 ? void 0 : user.companyId);
        // Fetch the company data
        const companyDoc = yield companyRef.get();
        // Check if company data exists
        const companyData = companyDoc.exists ? companyDoc.data() : null;
        return res.status(200).json({
            status: "success",
            code: 200,
            isAuthenticated: true,
            user: user,
            company: companyData,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Failed to fetch company data",
            error: error.message,
        });
    }
});
exports.checkAuth = checkAuth;
