"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteAnnouncement = exports.updateAnnouncement = exports.viewAnnouncement = exports.getAnnouncements = exports.createAnnouncement = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const zod_1 = require("zod");
const announcements_1 = require("../../../../types/announcements");
const checkWorkspaceOwnership = (workspaceId, user, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Fetch the workspace document from Firestore
        const workspaceSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        // Check if the workspace exists
        if (!workspaceSnapshot.exists) {
            res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
            return null; // Return null if the workspace doesn't exist
        }
        const workspaceData = workspaceSnapshot.data();
        // Check if the workspace belongs to the user's company
        if (workspaceData.companyId !== user.companyId) {
            res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have access to this workspace.",
            });
            return null; // Return null if the user doesn't have access
        }
        return workspaceData; // Return the workspace data if validation passes
    }
    catch (error) {
        res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
        return null; // Return null if there's an internal error
    }
});
const getAnnouncements = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const user = req.user;
        // Validate workspace ownership
        const workspaceData = yield checkWorkspaceOwnership(workspaceId, user, res);
        if (!workspaceData)
            return; // If the workspace check fails, it will already return the response.
        // Fetch all announcements for the specific workspace
        const snapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("announcements")
            .get();
        // Map the announcements to include the `createdBy` user data
        const announcements = yield Promise.all(snapshot.docs.map((doc) => __awaiter(void 0, void 0, void 0, function* () {
            var _a, _b, _c;
            const data = doc.data();
            // Fetch the creator's user data from Firestore
            const userSnapshot = yield firebase_admin_1.adminDb
                .collection("companies")
                .doc(user.companyId)
                .collection("users")
                .doc(data.createdBy)
                .get();
            const createdByUser = userSnapshot.exists
                ? {
                    id: userSnapshot.id,
                    name: ((_a = userSnapshot.data()) === null || _a === void 0 ? void 0 : _a.name) || "Unknown User",
                    email: ((_b = userSnapshot.data()) === null || _b === void 0 ? void 0 : _b.email) || "Unknown Email",
                    photoURL: ((_c = userSnapshot.data()) === null || _c === void 0 ? void 0 : _c.photoURL) || null,
                }
                : {
                    id: data.createdBy,
                    name: "Unknown User",
                    email: "Unknown Email",
                    photoURL: null,
                };
            return Object.assign(Object.assign({}, data), { id: doc.id, createdBy: createdByUser });
        })));
        // Filter announcements based on user role
        let filteredAnnouncements = announcements;
        if (user.role !== "superadmin" && user.role !== "admin") {
            filteredAnnouncements = announcements.filter((announcement) => {
                return (announcement.sharedToRole === "allUsers" ||
                    announcement.sharedToRole === user.role ||
                    (announcement.specificUsers &&
                        announcement.specificUsers.includes(user.id)));
            });
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            data: filteredAnnouncements,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.getAnnouncements = getAnnouncements;
const viewAnnouncement = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c;
    try {
        const { workspaceId, announcementId } = req.params;
        const user = req.user;
        // Validate workspace ownership
        const workspaceData = yield checkWorkspaceOwnership(workspaceId, user, res);
        if (!workspaceData)
            return;
        const announcementRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("announcements")
            .doc(announcementId);
        const announcementDoc = yield announcementRef.get();
        // Check if the announcement exists
        if (!announcementDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Announcement not found.",
            });
        }
        const announcementData = announcementDoc.data();
        // Fetch the creator's user data from Firestore
        const userSnapshot = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users")
            .doc(announcementData.createdBy)
            .get();
        const createdByUser = userSnapshot.exists
            ? {
                id: userSnapshot.id,
                name: ((_a = userSnapshot.data()) === null || _a === void 0 ? void 0 : _a.name) || "Unknown User",
                email: ((_b = userSnapshot.data()) === null || _b === void 0 ? void 0 : _b.email) || "Unknown Email",
                photoURL: ((_c = userSnapshot.data()) === null || _c === void 0 ? void 0 : _c.photoURL) || null,
            }
            : {
                id: announcementData.createdBy,
                name: "Unknown User",
                email: "Unknown Email",
                photoURL: null,
            };
        // Include detailed user data in the announcement
        const detailedAnnouncement = Object.assign(Object.assign({}, announcementData), { id: announcementId, createdBy: createdByUser });
        return res.status(200).json({
            status: "success",
            code: 200,
            data: detailedAnnouncement,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewAnnouncement = viewAnnouncement;
const createAnnouncement = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId } = req.params;
        const user = req.user;
        if (!user || (user.role !== "superadmin" && user.role !== "admin")) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Forbidden. Only admins can create announcements.",
            });
        }
        const parsedBody = announcements_1.createAnnouncementSchema.parse(req.body);
        // Validate workspace ownership
        const workspaceData = yield checkWorkspaceOwnership(workspaceId, user, res);
        if (!workspaceData)
            return;
        // Generate new announcement ID
        const docRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("announcements")
            .doc();
        const announcement = {
            id: docRef.id,
            title: parsedBody.title,
            label: parsedBody.label,
            description: parsedBody.description,
            sharedToRole: parsedBody.sharedToRole,
            specificUsers: parsedBody.specificUsers || [],
            createdBy: user.id,
            createdAt: new Date().toISOString(),
        };
        // Save the new announcement
        yield docRef.set(announcement);
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "Announcement created successfully",
            data: announcement,
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.createAnnouncement = createAnnouncement;
const updateAnnouncement = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, announcementId } = req.params;
        const user = req.user;
        const parsedBody = announcements_1.updateAnnouncementSchema.parse(req.body);
        // Validate workspace ownership
        const workspaceData = yield checkWorkspaceOwnership(workspaceId, user, res);
        if (!workspaceData)
            return;
        const announcementRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("announcements")
            .doc(announcementId);
        const announcementDoc = yield announcementRef.get();
        if (!announcementDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Announcement not found.",
            });
        }
        const announcementData = announcementDoc.data();
        // Superadmin can update any announcement, admin can only update their own
        if (user.role !== "superadmin" &&
            (announcementData === null || announcementData === void 0 ? void 0 : announcementData.createdBy) !== user.id) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Forbidden. You can only update your own announcements.",
            });
        }
        const updatedData = Object.assign(Object.assign(Object.assign({}, announcementData), parsedBody), { updatedAt: new Date().toISOString() });
        yield announcementRef.update(updatedData);
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Announcement updated successfully",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.updateAnnouncement = updateAnnouncement;
const deleteAnnouncement = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { workspaceId, announcementId } = req.params;
        const user = req.user;
        // Validate workspace ownership
        const workspaceData = yield checkWorkspaceOwnership(workspaceId, user, res);
        if (!workspaceData)
            return;
        const announcementRef = firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("announcements")
            .doc(announcementId);
        const announcementDoc = yield announcementRef.get();
        if (!announcementDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Announcement not found.",
            });
        }
        const announcementData = announcementDoc.data();
        // Superadmin can delete any announcement, admin can delete only their own
        if (user.role !== "superadmin" &&
            (announcementData === null || announcementData === void 0 ? void 0 : announcementData.createdBy) !== user.id) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Forbidden. You can only delete your own announcements.",
            });
        }
        yield announcementRef.delete();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Announcement deleted successfully",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteAnnouncement = deleteAnnouncement;
